/*
 * ads131m04.c
 *
 *  Created on: Feb 12, 2026
 *      Author: c6h6
 */

#include "ads131m04.h"


// ---------- GPIO helpers ----------
static inline void cs_low(void)  { HAL_GPIO_WritePin(ADS_CS_GPIO_PORT, ADS_CS_GPIO_PIN, GPIO_PIN_RESET); }
static inline void cs_high(void) { HAL_GPIO_WritePin(ADS_CS_GPIO_PORT, ADS_CS_GPIO_PIN, GPIO_PIN_SET);   }

static inline void rst_low(void)  { HAL_GPIO_WritePin(ADS_RESET_GPIO_PORT, ADS_RESET_GPIO_PIN, GPIO_PIN_RESET); }
static inline void rst_high(void) { HAL_GPIO_WritePin(ADS_RESET_GPIO_PORT, ADS_RESET_GPIO_PIN, GPIO_PIN_SET);   }

// ---------- 24-bit helpers ----------
static inline uint32_t be24_to_u32(const uint8_t b[3])
{
    return ((uint32_t)b[0] << 16) | ((uint32_t)b[1] << 8) | (uint32_t)b[2];
}

// 24-bit signed sign-extend
static inline int32_t sign_extend_24(uint32_t x24)
{
    if (x24 & 0x800000UL) return (int32_t)(x24 | 0xFF000000UL);
    return (int32_t)x24;
}

// Put 16-bit command into first 24-bit word: [cmd_hi][cmd_lo][0x00]
static inline void cmd16_to_word0_24(uint16_t cmd, uint8_t out3[3])
{
    out3[0] = (uint8_t)(cmd >> 8);
    out3[1] = (uint8_t)(cmd & 0xFF);
    out3[2] = 0x00;
}

// Extract register 16-bit from response word0 (CRC OFF):
// Based on your capture: rx[0]=0x02 rx[1]=0x87 rx[2]=0x80 => reg16 is 0x0287 (word0[23:8])
static inline uint16_t w0_reg16_from_rx(const uint8_t rx[ADS_FRAME_BYTES])
{
    return (uint16_t)((uint16_t)rx[0] << 8 | rx[1]);
}

// Optional: response/status byte (word0[7:0]) if you want it
static inline uint8_t w0_resp8_from_rx(const uint8_t rx[ADS_FRAME_BYTES])
{
    return rx[2];
}
static uint16_t crc_ccitt_1021(const uint8_t *data, uint32_t len, uint16_t init)
{
    uint16_t crc = init;
    for (uint32_t i = 0; i < len; i++) {
        crc ^= (uint16_t)data[i] << 8;
        for (int b = 0; b < 8; b++) {
            crc = (crc & 0x8000) ? (uint16_t)((crc << 1) ^ 0x1021) : (uint16_t)(crc << 1);
        }
    }
    return crc;
}
static inline void ads_cs_gap_us(uint32_t us)
{
    // 아주 대충이라도 1~2us 보장 (DWT 쓰면 더 정확)
    for (volatile uint32_t i = 0; i < us * 50; i++) __NOP();
}
// ===========================
// Public API
// ===========================
void ads_hw_init_pins_idle(void)
{
    cs_high();
    rst_high();
}

void ads_reset_pulse(uint32_t delay_ms)
{
    rst_low();
    HAL_Delay(delay_ms);
    rst_high();
    HAL_Delay(delay_ms);
}

HAL_StatusTypeDef ads_clkin_start(void)
{
    return (HAL_TIM_PWM_Start(&ADS_CLK_TIM_HANDLE, ADS_CLK_TIM_CHANNEL) == HAL_OK) ? HAL_OK : HAL_ERROR;
}

HAL_StatusTypeDef ads_clkin_stop(void)
{
    return (HAL_TIM_PWM_Stop(&ADS_CLK_TIM_HANDLE, ADS_CLK_TIM_CHANNEL) == HAL_OK) ? HAL_OK : HAL_ERROR;
}
static void ads_build_tx_frame(uint16_t cmd, const uint8_t *word1_24, uint8_t tx[18])
{
    for (int i = 0; i < 18; i++) tx[i] = 0x00;

#if ADS_USE_CRC == 0
    // CRC OFF: word0 = [cmd_hi][cmd_lo][0x00]
    tx[0] = (uint8_t)(cmd >> 8);
    tx[1] = (uint8_t)(cmd & 0xFF);
    tx[2] = 0x00;
#else
    // CRC ON: DIN begins with CMD16 + CRC16 (immediately)
    tx[0] = (uint8_t)(cmd >> 8);
    tx[1] = (uint8_t)(cmd & 0xFF);
    uint16_t crc = crc_ccitt_1021(&tx[0], 2, ADS_CRC_INIT);
    tx[2] = (uint8_t)(crc >> 8);
    tx[3] = (uint8_t)(crc & 0xFF);
#endif

    // optional word1 data (WREG)
    if (word1_24) {
        tx[3] = word1_24[0];
        tx[4] = word1_24[1];
        tx[5] = word1_24[2];
    }
}
// Core: 18-byte frame transfer
HAL_StatusTypeDef ads_xfer_frame(uint16_t cmd_word,
                                 const uint8_t *tx_word1_24, // 3 bytes if not NULL
                                 uint8_t rx_out[ADS_FRAME_BYTES])
{
    uint8_t tx[ADS_FRAME_BYTES], rx[ADS_FRAME_BYTES];

    ads_build_tx_frame(cmd_word, tx_word1_24, tx);

    cs_low();
    HAL_StatusTypeDef st = HAL_SPI_TransmitReceive(&ADS_SPI_HANDLE, tx, rx, 18, ADS_SPI_TIMEOUT_MS);
    cs_high();
    ads_cs_gap_us(5);
    if (st != HAL_OK) return st;

    if (rx_out) for (int i = 0; i < 18; i++) rx_out[i] = rx[i];
    return HAL_OK;
}


static HAL_StatusTypeDef ads_wait_drdy_fall(uint32_t timeout_ms)
{
    // 간단 폴링 버전: DRDY가 high->low 되는 순간을 기다림
    uint32_t t0 = HAL_GetTick();
    GPIO_PinState prev = HAL_GPIO_ReadPin(ADS_DRDY_GPIO_PORT, ADS_DRDY_GPIO_PIN);

    while ((HAL_GetTick() - t0) < timeout_ms) {
        GPIO_PinState now = HAL_GPIO_ReadPin(ADS_DRDY_GPIO_PORT, ADS_DRDY_GPIO_PIN);
        if (prev == GPIO_PIN_SET && now == GPIO_PIN_RESET) return HAL_OK;
        prev = now;
    }
    return HAL_TIMEOUT;
}

HAL_StatusTypeDef ads_read_channels(int32_t ch_out[4])
{
    uint8_t rx[ADS_FRAME_BYTES];
    if (ads_wait_drdy_fall(ADS_DRDY_TIMEOUT_MS) != HAL_OK) return HAL_TIMEOUT;
    HAL_StatusTypeDef st = ads_xfer_frame(0x0000, NULL, rx);
    if (st != HAL_OK) return st;

    // word1..word4: CH0..CH3 (24-bit signed)
    for (int i = 0; i < 4; i++) {
        uint32_t w = be24_to_u32(&rx[(1 + i) * 3]);
        ch_out[i] = sign_extend_24(w);
    }
    return HAL_OK;
}

// Single register read (count=1): response comes in NEXT frame's word0
HAL_StatusTypeDef ads_read_reg(uint8_t reg, uint16_t *out)
{
    uint8_t rx[18];
    uint16_t cmd = ADS_CMD_RREG(reg, 1);

    // Frame N: DRDY fall 기다렸다가 RREG
    if (ads_wait_drdy_fall(ADS_DRDY_TIMEOUT_MS) != HAL_OK) return HAL_TIMEOUT;
    HAL_StatusTypeDef st = ads_xfer_frame(cmd, NULL, rx);
    if (st != HAL_OK) return st;

    // Frame N+1: 다음 DRDY fall 기다렸다가 NOP (여기서 레지스터 데이터)
    if (ads_wait_drdy_fall(ADS_DRDY_TIMEOUT_MS) != HAL_OK) return HAL_TIMEOUT;
    st = ads_xfer_frame(0x0000, NULL, rx);
    if (st != HAL_OK) return st;

    // 레지스터 값은 rx[0..1]에서
    if (out) *out = (uint16_t)((uint16_t)rx[0] << 8 | rx[1]);
    return HAL_OK;
}

// Single register write (count=1):
// Typically: Frame N: WREG + data (data in word1), Frame N+1: ack/response (device-dependent)
HAL_StatusTypeDef ads_write_reg(uint8_t reg, uint16_t data, bool *ok)
{
    uint8_t rx[ADS_FRAME_BYTES];

    // word1 data 24-bit: [data_hi][data_lo][0x00]
    uint8_t w1[3] = {(uint8_t)(data >> 8), (uint8_t)(data & 0xFF), 0x00};

    uint16_t cmd = ADS_CMD_WREG(reg, 1);

    // Frame N: send WREG + data
    HAL_StatusTypeDef st = ads_xfer_frame(cmd, w1, rx);
    if (st != HAL_OK) return st;

    // Frame N+1: NOP to get response/ack (many TI devices do this one-frame-later)
    st = ads_xfer_frame(0x0000, NULL, rx);
    if (st != HAL_OK) return st;

    // CRC OFF일 때도 response word0의 상위16에 "응답 코드"가 오는 경우가 있어.
    // 다만 “ack 포맷”은 데이터시트의 WREG 응답 정의를 봐야 100% 확정.
    // 여기선 최소한 "원하는 레지스터를 다시 읽어서 검증"하는 방식이 가장 확실함.
    if (ok) {
        uint16_t rb = 0;
        if (ads_read_reg(reg, &rb) == HAL_OK) *ok = (rb == data);
        else *ok = false;
    }

    return HAL_OK;
}

HAL_StatusTypeDef ads_set_gain(uint8_t log2Gain0,
                               uint8_t log2Gain1,
                               uint8_t log2Gain2,
                               uint8_t log2Gain3,
                               bool *ok)
{
    uint16_t gain = 0;
    gain  = (uint16_t)((log2Gain3 & 0x0F) << 4);
    gain |= (uint16_t)(log2Gain2 & 0x0F);
    gain <<= 8;
    gain |= (uint16_t)((log2Gain1 & 0x0F) << 4);
    gain |= (uint16_t)(log2Gain0 & 0x0F);

    return ads_write_reg(ADS131M04_REG_GAIN1, gain, ok);
}

