#include "modbus.h"
#include "modbus_rtu.h"
#include <string.h>
#include <math.h>

// 내부 함수 선언
static bool parse_frame_from_ring_buffer(Modbus_t *modbus);

#ifdef ASCII_MODE
// 초기화 함수
void Modbus_Init(Modbus_t *modbus, UART_HandleTypeDef *huart, uint8_t slaveId, ModbusAscii_t *ascii) {
#else
void Modbus_Init(Modbus_t *modbus, UART_HandleTypeDef *huart, uint8_t slaveId) {
#endif
    // 기본 설정
    modbus->huart = huart;
    modbus->slaveId = slaveId;
    
    // 상태 초기화
    modbus->status.rxStatus = RX_IDLE;
    modbus->status.txStatus = TX_IDLE;
    modbus->status.parseStatus = PARSE_OK;
    modbus->status.bufferStatus = RINGBUF_OK;
    modbus->status.errorCounter = 0;
    modbus->status.lastErrorCode = 0;
    modbus->status.currentRxMode = MODBUS_MODE_RTU;
    modbus->status.currentParseMode = MODBUS_MODE_RTU;
    
    // 버퍼 초기화
    memset(&modbus->buffers.rxFrameBuffer, 0, sizeof(Frame_t));
    memset(&modbus->buffers.currentFrame, 0, sizeof(Frame_t));
    memset(&modbus->buffers.txFrameBuffer, 0, sizeof(Frame_t));
    memset(modbus->buffers.rxRingFrameBuffer, 0, sizeof(modbus->buffers.rxRingFrameBuffer));
    modbus->buffers.rxWriteIndex = 0;
    modbus->buffers.rxReadIndex = 0;
    modbus->buffers.frameReadyFlag = false;
    modbus->buffers.firstDataFlag = true;
    
    // 레지스터 초기화
    memset(&modbus->registers, 0, sizeof(ModbusRegisters_t));
    
    #ifdef ASCII_MODE
    modbus->ascii = ascii;  // ASCII 모듈 포인터 설정
    // ASCII 모듈 초기화
    if (modbus->ascii != NULL) {
        modbus->ascii->timeoutCounter = 0;
        modbus->ascii->receptionActive = false;
    }
    #endif
    
    // IT 수신 시작작
    HAL_UART_Receive_IT(modbus->huart, modbus->buffers.rxFrameBuffer.data, 1);
    
    // IDLE 라인 감지 활성화
    __HAL_UART_ENABLE_IT(modbus->huart, UART_IT_IDLE);
}

// 메인 처리 함수
void Modbus_Process(Modbus_t *modbus) {
    if (modbus->buffers.frameReadyFlag) {
        if (parse_frame_from_ring_buffer(modbus)) {
            // 리스폰 데이터 준비
            Modbus_PrepareResponseData(modbus);
            
            // 리스폰 데이터 전송
            Modbus_SendResponseData(modbus);
        }
        modbus->buffers.frameReadyFlag = false;
    }
}

// UART IDLE 콜백
void Modbus_UARTIdleCallback(Modbus_t *modbus, UART_HandleTypeDef *huart) {
    if (huart == modbus->huart) {
        if (modbus->buffers.firstDataFlag == false) {
            // RTU 모드일 때만 IDLE 감지로 프레임 완료 처리
            if (modbus->status.currentRxMode == MODBUS_MODE_RTU) {
                ModbusRTU_RxFrameComplete(modbus, huart);
            }
            if (modbus->status.currentRxMode != MODBUS_MODE_ASCII) {
                // 버퍼 초기화
                memset(modbus->buffers.rxFrameBuffer.data, 0, BUFFER_SIZE);
                // 버퍼 인덱스 초기화
                modbus->buffers.rxFrameBuffer.length = 0;

                modbus->buffers.firstDataFlag = true;
                
                HAL_UART_Abort_IT(modbus->huart);
                HAL_UART_Receive_IT(modbus->huart, modbus->buffers.rxFrameBuffer.data, 1);
            
            }
        }
    }
}

// UART 수신 콜백
void Modbus_UARTRxCallback(Modbus_t *modbus, UART_HandleTypeDef *huart) {
    if (huart == modbus->huart) {
        if (modbus->buffers.firstDataFlag == true) {
            // RTU 프레임의 첫 바이트는 슬레이브 ID
            if (modbus->buffers.rxFrameBuffer.data[0] == modbus->slaveId) {
                modbus->status.currentRxMode = MODBUS_MODE_RTU;
                modbus->buffers.firstDataFlag = false;
                modbus->buffers.rxFrameBuffer.length++; // 첫 바이트 저장 완료 → 다음 인덱스로 이동
            }
            else {
                modbus->status.currentRxMode = MODBUS_MODE_ETC;
                modbus->buffers.firstDataFlag = false;
                modbus->buffers.rxFrameBuffer.length++; // 첫 바이트 저장 완료 → 다음 인덱스로 이동
                // modbus->buffers.firstDataFlag = true;
                // modbus->buffers.rxFrameBuffer.length = 0;
                // modbus->status.currentRxMode = MODBUS_MODE_ETC;

                // HAL_UART_Abort_IT(modbus->huart); // 혹시 모를 남은 DMA/IT 중단
                // HAL_UART_Receive_IT(modbus->huart, &modbus->buffers.rxFrameBuffer.data[0], 1);
                return; 
            }
        }
        else if(modbus->buffers.firstDataFlag == false){
            modbus->buffers.rxFrameBuffer.length++; // 다음 바이트 저장
        }

        // modbus->buffers.rxFrameBuffer.length++;
        // 다음 바이트 수신을 위해 인터럽트 다시 시작
        if(modbus->buffers.rxFrameBuffer.length < BUFFER_SIZE) {
            if (HAL_UART_Receive_IT(modbus->huart, &modbus->buffers.rxFrameBuffer.data[modbus->buffers.rxFrameBuffer.length], 1) != HAL_OK) {
                #ifdef DEBUG_MODE
                modbus->status.rxStatus = RX_ERROR;
                modbus->registers.holdingRegisters[DEBUG_RX_STATUS_ADDR] = modbus->status.rxStatus;
                modbus->status.errorCounter++;
                modbus->registers.holdingRegisters[DEBUG_ERROR_COUNT_ADDR] = modbus->status.errorCounter;
                #endif
                // Error_Handler();
            }
        }
        else {
            // 버퍼 오버플로우 처리
            #ifdef DEBUG_MODE
            modbus->status.rxStatus = RX_ERROR_OVERFLOW;
            modbus->registers.holdingRegisters[DEBUG_RX_STATUS_ADDR] = modbus->status.rxStatus;
            modbus->status.lastErrorCode = RX_ERROR_OVERFLOW;
            modbus->status.errorCounter++;
            modbus->registers.holdingRegisters[DEBUG_ERROR_COUNT_ADDR] = modbus->status.errorCounter;
            #endif
            // 버퍼 초기화
            memset(&modbus->buffers.rxFrameBuffer, 0, sizeof(Frame_t));
            modbus->buffers.rxFrameBuffer.length = 0;
            modbus->buffers.firstDataFlag = true;
            
            // 수신 재시작
            HAL_UART_Receive_IT(modbus->huart, &modbus->buffers.rxFrameBuffer.data[0], 1);
        }

        
    }
}

// UART 송신 완료 콜백
void Modbus_UARTTxCallback(Modbus_t *modbus, UART_HandleTypeDef *huart) {
    if (huart == modbus->huart) {
        #ifdef DEBUG_MODE
        modbus->status.txStatus = TX_COMPLETE;
        modbus->registers.holdingRegisters[DEBUG_TX_STATUS_ADDR] = modbus->status.txStatus;
        
        // 송신 완료 후 추가 작업 수행
        memset(modbus->buffers.txFrameBuffer.data, 0, BUFFER_SIZE);
        modbus->buffers.txFrameBuffer.length = 0;

        modbus->status.txStatus = TX_IDLE;
        modbus->registers.holdingRegisters[DEBUG_TX_STATUS_ADDR] = modbus->status.txStatus;
        #else
         // 송신 완료 후 추가 작업 수행
         memset(modbus->buffers.txFrameBuffer.data, 0, BUFFER_SIZE);
         modbus->buffers.txFrameBuffer.length = 0;
        #endif
    }
}

// 레지스터 접근 함수
uint16_t Modbus_GetHoldingRegister(Modbus_t *modbus, uint16_t address) {
    if (address < HOLDING_REGISTERS_SIZE) {
        return modbus->registers.holdingRegisters[address];
    }
    return 0;
}

void Modbus_SetHoldingRegister(Modbus_t *modbus, uint16_t address, uint16_t value) {
    if (address < HOLDING_REGISTERS_SIZE) {
        modbus->registers.holdingRegisters[address] = value;
    }
}

uint16_t Modbus_GetInputRegister(Modbus_t *modbus, uint16_t address) {
    if (address < INPUT_REGISTERS_SIZE) {
        return modbus->registers.inputRegisters[address];
    }
    return 0;
}

void Modbus_SetInputRegister(Modbus_t *modbus, uint16_t address, uint16_t value) {
    if (address < INPUT_REGISTERS_SIZE) {
        modbus->registers.inputRegisters[address] = value;
    }
}

uint8_t Modbus_GetCoil(Modbus_t *modbus, uint16_t address) {
    if (address < COILS_SIZE) {
        return modbus->registers.coils[address];
    }
    return 0;
}

void Modbus_SetCoil(Modbus_t *modbus, uint16_t address, uint8_t value) {
    if (address < COILS_SIZE) {
        modbus->registers.coils[address] = value ? 1 : 0; // 0이 아닌 값은 모두 1로 저장
    }
}

uint8_t Modbus_GetDiscreteInput(Modbus_t *modbus, uint16_t address) {
    if (address < DISCRETE_INPUTS_SIZE) {
        return modbus->registers.discreteInputs[address];
    }
    return 0;
}

void Modbus_SetDiscreteInput(Modbus_t *modbus, uint16_t address, uint8_t value) {
    if (address < DISCRETE_INPUTS_SIZE) {
        modbus->registers.discreteInputs[address] = value ? 1 : 0; // 0이 아닌 값은 모두 1로 저장
    }
}

// 예외 응답 생성 함수
void Modbus_CreateExceptionResponse(Modbus_t *modbus, uint8_t functionCode, uint8_t exceptionCode) {
    modbus->buffers.txFrameBuffer.data[0] = modbus->slaveId;
    modbus->buffers.txFrameBuffer.data[1] = functionCode | 0x80;  // 예외 응답 플래그 설정
    modbus->buffers.txFrameBuffer.data[2] = exceptionCode;
    
    // RTU 모드인 경우 CRC 추가
    if (modbus->status.currentParseMode == MODBUS_MODE_RTU) {
        uint16_t crc = ModbusRTU_CalculateCRC(modbus->buffers.txFrameBuffer.data, 3);
        modbus->buffers.txFrameBuffer.data[3] = crc & 0xFF;
        modbus->buffers.txFrameBuffer.data[4] = (crc >> 8) & 0xFF;
        modbus->buffers.txFrameBuffer.length = 5;
    } else {
        // ASCII 모드인 경우 LRC 계산은 전송 전에 별도로 처리됨
        modbus->buffers.txFrameBuffer.length = 3;
    }
    
    #ifdef DEBUG_MODE
    // 디버깅 정보 업데이트
    modbus->registers.holdingRegisters[DEBUG_TX_LENGTH_ADDR] = modbus->buffers.txFrameBuffer.length;
    modbus->status.parseStatus = PARSE_ERROR_FUNC;
    modbus->registers.holdingRegisters[DEBUG_PARSE_STATUS_ADDR] = modbus->status.parseStatus;
    #endif
}
// 슬레이브 아이디 설정 함수
void Modbus_SetSlaveId(Modbus_t *modbus, uint8_t slaveId) {
    modbus->slaveId = slaveId;
}
// 응답 데이터 준비 함수
void Modbus_PrepareResponseData(Modbus_t *modbus) {
    uint8_t functionCode = modbus->buffers.currentFrame.data[1];
    uint16_t startAddress, quantity, value;
    uint8_t byteCount;
    uint16_t i, crc;
    
    switch (functionCode) {
        case MODBUS_FC_READ_HOLDING_REGISTERS:
            startAddress = (modbus->buffers.currentFrame.data[2] << 8) | modbus->buffers.currentFrame.data[3];
            quantity = (modbus->buffers.currentFrame.data[4] << 8) | modbus->buffers.currentFrame.data[5];
            
            // 주소 범위 확인
            if (startAddress + quantity <= HOLDING_REGISTERS_SIZE && quantity > 0 && quantity <= 125) {
                // 응답 헤더 구성
                modbus->buffers.txFrameBuffer.data[0] = modbus->slaveId;
                modbus->buffers.txFrameBuffer.data[1] = MODBUS_FC_READ_HOLDING_REGISTERS;
                modbus->buffers.txFrameBuffer.data[2] = quantity * 2;  // 바이트 수
                
                // 데이터 구성
                for (i = 0; i < quantity; i++) {
                    modbus->buffers.txFrameBuffer.data[3 + i * 2] = (modbus->registers.holdingRegisters[startAddress + i] >> 8) & 0xFF;  // 상위 바이트
                    modbus->buffers.txFrameBuffer.data[4 + i * 2] = modbus->registers.holdingRegisters[startAddress + i] & 0xFF;         // 하위 바이트
                }
                
                if (modbus->status.currentParseMode == MODBUS_MODE_RTU) {
                    // CRC 추가
                    crc = ModbusRTU_CalculateCRC(modbus->buffers.txFrameBuffer.data, 3 + quantity * 2);
                    modbus->buffers.txFrameBuffer.data[3 + quantity * 2] = crc & 0xFF;
                    modbus->buffers.txFrameBuffer.data[4 + quantity * 2] = (crc >> 8) & 0xFF;
                    
                    modbus->buffers.txFrameBuffer.length = 5 + quantity * 2;
                } else {
                    modbus->buffers.txFrameBuffer.length = 3 + quantity * 2;
                }
            } else {
                Modbus_CreateExceptionResponse(modbus, MODBUS_FC_READ_HOLDING_REGISTERS, MODBUS_EXCEPTION_ILLEGAL_DATA_ADDRESS);
            }
            break;
            
            case MODBUS_FC_READ_INPUT_REGISTERS:
            startAddress = (modbus->buffers.currentFrame.data[2] << 8) | modbus->buffers.currentFrame.data[3];
            quantity = (modbus->buffers.currentFrame.data[4] << 8) | modbus->buffers.currentFrame.data[5];
            
            // 주소 범위 확인
            if (startAddress + quantity <= INPUT_REGISTERS_SIZE && quantity > 0 && quantity <= 125) {
                // 응답 헤더 구성
                modbus->buffers.txFrameBuffer.data[0] = modbus->slaveId;
                modbus->buffers.txFrameBuffer.data[1] = MODBUS_FC_READ_INPUT_REGISTERS;
                modbus->buffers.txFrameBuffer.data[2] = quantity * 2;  // 바이트 수
                
                // 데이터 구성
                for (i = 0; i < quantity; i++) {
                    modbus->buffers.txFrameBuffer.data[3 + i * 2] = (modbus->registers.inputRegisters[startAddress + i] >> 8) & 0xFF;  // 상위 바이트
                    modbus->buffers.txFrameBuffer.data[4 + i * 2] = modbus->registers.inputRegisters[startAddress + i] & 0xFF;         // 하위 바이트
                }
                
                if (modbus->status.currentParseMode == MODBUS_MODE_RTU) {
                    // CRC 추가
                    crc = ModbusRTU_CalculateCRC(modbus->buffers.txFrameBuffer.data, 3 + quantity * 2);
                    modbus->buffers.txFrameBuffer.data[3 + quantity * 2] = crc & 0xFF;
                    modbus->buffers.txFrameBuffer.data[4 + quantity * 2] = (crc >> 8) & 0xFF;
                    
                    modbus->buffers.txFrameBuffer.length = 5 + quantity * 2;
                } else {
                    modbus->buffers.txFrameBuffer.length = 3 + quantity * 2;
                }
            } else {
                Modbus_CreateExceptionResponse(modbus, MODBUS_FC_READ_INPUT_REGISTERS, MODBUS_EXCEPTION_ILLEGAL_DATA_ADDRESS);
            }
            break;
            
        case MODBUS_FC_READ_COILS:
            startAddress = (modbus->buffers.currentFrame.data[2] << 8) | modbus->buffers.currentFrame.data[3];
            quantity = (modbus->buffers.currentFrame.data[4] << 8) | modbus->buffers.currentFrame.data[5];
            
            // 주소 범위 확인
            if (startAddress + quantity <= COILS_SIZE && quantity > 0 && quantity <= 2000) {
                // 응답 헤더 구성
                modbus->buffers.txFrameBuffer.data[0] = modbus->slaveId;
                modbus->buffers.txFrameBuffer.data[1] = MODBUS_FC_READ_COILS;
                
                // 바이트 수 계산 (8코일당 1바이트, 나머지가 있으면 1바이트 추가)
                byteCount = (quantity + 7) / 8;
                modbus->buffers.txFrameBuffer.data[2] = byteCount;
                
                // 데이터 초기화
                memset(&modbus->buffers.txFrameBuffer.data[3], 0, byteCount);
                
                // 데이터 구성
                for (i = 0; i < quantity; i++) {
                    if (modbus->registers.coils[startAddress + i]) {
                        modbus->buffers.txFrameBuffer.data[3 + i/8] |= (1 << (i % 8));
                    }
                }
                
                if (modbus->status.currentParseMode == MODBUS_MODE_RTU) {
                    // CRC 추가
                    crc = ModbusRTU_CalculateCRC(modbus->buffers.txFrameBuffer.data, 3 + byteCount);
                    modbus->buffers.txFrameBuffer.data[3 + byteCount] = crc & 0xFF;
                    modbus->buffers.txFrameBuffer.data[4 + byteCount] = (crc >> 8) & 0xFF;
                    
                    modbus->buffers.txFrameBuffer.length = 5 + byteCount;
                } else {
                    modbus->buffers.txFrameBuffer.length = 3 + byteCount;
                }
            } else {
                Modbus_CreateExceptionResponse(modbus, MODBUS_FC_READ_COILS, MODBUS_EXCEPTION_ILLEGAL_DATA_ADDRESS);
            }
            break;
            
        case MODBUS_FC_READ_DISCRETE_INPUTS:
            startAddress = (modbus->buffers.currentFrame.data[2] << 8) | modbus->buffers.currentFrame.data[3];
            quantity = (modbus->buffers.currentFrame.data[4] << 8) | modbus->buffers.currentFrame.data[5];
            
            // 주소 범위 확인
            if (startAddress + quantity <= DISCRETE_INPUTS_SIZE && quantity > 0 && quantity <= 2000) {
                // 응답 헤더 구성
                modbus->buffers.txFrameBuffer.data[0] = modbus->slaveId;
                modbus->buffers.txFrameBuffer.data[1] = MODBUS_FC_READ_DISCRETE_INPUTS;
                
                // 바이트 수 계산 (8입력당 1바이트, 나머지가 있으면 1바이트 추가)
                byteCount = (quantity + 7) / 8;
                modbus->buffers.txFrameBuffer.data[2] = byteCount;
                
                // 데이터 초기화
                memset(&modbus->buffers.txFrameBuffer.data[3], 0, byteCount);
                
                // 데이터 구성
                for (i = 0; i < quantity; i++) {
                    if (modbus->registers.discreteInputs[startAddress + i]) {
                        modbus->buffers.txFrameBuffer.data[3 + i/8] |= (1 << (i % 8));
                    }
                }
                
                if (modbus->status.currentParseMode == MODBUS_MODE_RTU) {
                    // CRC 추가
                    crc = ModbusRTU_CalculateCRC(modbus->buffers.txFrameBuffer.data, 3 + byteCount);
                    modbus->buffers.txFrameBuffer.data[3 + byteCount] = crc & 0xFF;
                    modbus->buffers.txFrameBuffer.data[4 + byteCount] = (crc >> 8) & 0xFF;
                    
                    modbus->buffers.txFrameBuffer.length = 5 + byteCount;
                } else {
                    modbus->buffers.txFrameBuffer.length = 3 + byteCount;
                }
            } else {
                Modbus_CreateExceptionResponse(modbus, MODBUS_FC_READ_DISCRETE_INPUTS, MODBUS_EXCEPTION_ILLEGAL_DATA_ADDRESS);
            }
            break;
            
        case MODBUS_FC_WRITE_SINGLE_REGISTER:
            startAddress = (modbus->buffers.currentFrame.data[2] << 8) | modbus->buffers.currentFrame.data[3];
            value = (modbus->buffers.currentFrame.data[4] << 8) | modbus->buffers.currentFrame.data[5];
            
            // 주소 범위 확인
            if (startAddress < HOLDING_REGISTERS_SIZE) {
                // 레지스터에 값 쓰기
                modbus->registers.holdingRegisters[startAddress] = value;
                
                // 요청과 동일한 응답 반환
                memcpy(modbus->buffers.txFrameBuffer.data, modbus->buffers.currentFrame.data, 6);
                
                if (modbus->status.currentParseMode == MODBUS_MODE_RTU) {
                    // CRC 추가
                    crc = ModbusRTU_CalculateCRC(modbus->buffers.txFrameBuffer.data, 6);
                    modbus->buffers.txFrameBuffer.data[6] = crc & 0xFF;
                    modbus->buffers.txFrameBuffer.data[7] = (crc >> 8) & 0xFF;
                    
                    modbus->buffers.txFrameBuffer.length = 8;
                } else {
                    modbus->buffers.txFrameBuffer.length = 6;
                }
            } else {
                Modbus_CreateExceptionResponse(modbus, MODBUS_FC_WRITE_SINGLE_REGISTER, MODBUS_EXCEPTION_ILLEGAL_DATA_ADDRESS);
            }
            break;
            
        case MODBUS_FC_WRITE_SINGLE_COIL:
            startAddress = (modbus->buffers.currentFrame.data[2] << 8) | modbus->buffers.currentFrame.data[3];
            value = (modbus->buffers.currentFrame.data[4] << 8) | modbus->buffers.currentFrame.data[5];
            
            // 주소 범위 및 값 확인 (0x0000=OFF, 0xFF00=ON)
            if (startAddress < COILS_SIZE && (value == 0x0000 || value == 0xFF00)) {
                // 코일에 값 쓰기
                modbus->registers.coils[startAddress] = (value == 0xFF00) ? 1 : 0;
                
                // 요청과 동일한 응답 반환
                memcpy(modbus->buffers.txFrameBuffer.data, modbus->buffers.currentFrame.data, 6);
                
                if (modbus->status.currentParseMode == MODBUS_MODE_RTU) {
                    // CRC 추가
                    crc = ModbusRTU_CalculateCRC(modbus->buffers.txFrameBuffer.data, 6);
                    modbus->buffers.txFrameBuffer.data[6] = crc & 0xFF;
                    modbus->buffers.txFrameBuffer.data[7] = (crc >> 8) & 0xFF;
                    
                    modbus->buffers.txFrameBuffer.length = 8;
                } else {
                    modbus->buffers.txFrameBuffer.length = 6;
                }
            } else {
                Modbus_CreateExceptionResponse(modbus, MODBUS_FC_WRITE_SINGLE_COIL, MODBUS_EXCEPTION_ILLEGAL_DATA_ADDRESS);
            }
            break;
            
        case MODBUS_FC_WRITE_MULTIPLE_REGISTERS:
            startAddress = (modbus->buffers.currentFrame.data[2] << 8) | modbus->buffers.currentFrame.data[3];
            quantity = (modbus->buffers.currentFrame.data[4] << 8) | modbus->buffers.currentFrame.data[5];
            byteCount = modbus->buffers.currentFrame.data[6];
            
            // 주소 범위 및 바이트 수 확인
            if (startAddress + quantity <= HOLDING_REGISTERS_SIZE && quantity > 0 && quantity <= 123 && byteCount == quantity * 2) {
                // 레지스터에 값 쓰기
                for (i = 0; i < quantity; i++) {
                    modbus->registers.holdingRegisters[startAddress + i] = 
                        (modbus->buffers.currentFrame.data[7 + i * 2] << 8) | 
                        modbus->buffers.currentFrame.data[8 + i * 2];
                }
                
                // 응답 구성
                modbus->buffers.txFrameBuffer.data[0] = modbus->slaveId;
                modbus->buffers.txFrameBuffer.data[1] = MODBUS_FC_WRITE_MULTIPLE_REGISTERS;
                modbus->buffers.txFrameBuffer.data[2] = modbus->buffers.currentFrame.data[2];  // 시작 주소 상위 바이트
                modbus->buffers.txFrameBuffer.data[3] = modbus->buffers.currentFrame.data[3];  // 시작 주소 하위 바이트
                modbus->buffers.txFrameBuffer.data[4] = modbus->buffers.currentFrame.data[4];  // 레지스터 개수 상위 바이트
                modbus->buffers.txFrameBuffer.data[5] = modbus->buffers.currentFrame.data[5];  // 레지스터 개수 하위 바이트
                
                if (modbus->status.currentParseMode == MODBUS_MODE_RTU) {
                    // CRC 추가
                    crc = ModbusRTU_CalculateCRC(modbus->buffers.txFrameBuffer.data, 6);
                    modbus->buffers.txFrameBuffer.data[6] = crc & 0xFF;
                    modbus->buffers.txFrameBuffer.data[7] = (crc >> 8) & 0xFF;
                    
                    modbus->buffers.txFrameBuffer.length = 8;
                } else {
                    modbus->buffers.txFrameBuffer.length = 6;
                }
            } else {
                Modbus_CreateExceptionResponse(modbus, MODBUS_FC_WRITE_MULTIPLE_REGISTERS, MODBUS_EXCEPTION_ILLEGAL_DATA_ADDRESS);
            }
            break;
            
        case MODBUS_FC_WRITE_MULTIPLE_COILS:
            startAddress = (modbus->buffers.currentFrame.data[2] << 8) | modbus->buffers.currentFrame.data[3];
            quantity = (modbus->buffers.currentFrame.data[4] << 8) | modbus->buffers.currentFrame.data[5];
            byteCount = modbus->buffers.currentFrame.data[6];
            
            // 주소 범위 및 바이트 수 확인
            if (startAddress + quantity <= COILS_SIZE && quantity > 0 && quantity <= 1968 && byteCount == (quantity + 7) / 8) {
                // 코일에 값 쓰기
                for (i = 0; i < quantity; i++) {
                    modbus->registers.coils[startAddress + i] = 
                        (modbus->buffers.currentFrame.data[7 + i/8] >> (i % 8)) & 0x01;
                }
                
                // 응답 구성
                modbus->buffers.txFrameBuffer.data[0] = modbus->slaveId;
                modbus->buffers.txFrameBuffer.data[1] = MODBUS_FC_WRITE_MULTIPLE_COILS;
                modbus->buffers.txFrameBuffer.data[2] = modbus->buffers.currentFrame.data[2];  // 시작 주소 상위 바이트
                modbus->buffers.txFrameBuffer.data[3] = modbus->buffers.currentFrame.data[3];  // 시작 주소 하위 바이트
                modbus->buffers.txFrameBuffer.data[4] = modbus->buffers.currentFrame.data[4];  // 코일 개수 상위 바이트
                modbus->buffers.txFrameBuffer.data[5] = modbus->buffers.currentFrame.data[5];  // 코일 개수 하위 바이트
                
                if (modbus->status.currentParseMode == MODBUS_MODE_RTU) {
                    // CRC 추가
                    crc = ModbusRTU_CalculateCRC(modbus->buffers.txFrameBuffer.data, 6);
                    modbus->buffers.txFrameBuffer.data[6] = crc & 0xFF;
                    modbus->buffers.txFrameBuffer.data[7] = (crc >> 8) & 0xFF;
                    
                    modbus->buffers.txFrameBuffer.length = 8;
                } else {
                    modbus->buffers.txFrameBuffer.length = 6;
                }
            } else {
                Modbus_CreateExceptionResponse(modbus, MODBUS_FC_WRITE_MULTIPLE_COILS, MODBUS_EXCEPTION_ILLEGAL_DATA_ADDRESS);
            }
            break;
        
        default:
            Modbus_CreateExceptionResponse(modbus, functionCode, MODBUS_EXCEPTION_ILLEGAL_FUNCTION);
            break;
    }
    
    #ifdef DEBUG_MODE
    // 응답 길이 저장 (디버깅용)
    modbus->registers.holdingRegisters[DEBUG_TX_LENGTH_ADDR] = modbus->buffers.txFrameBuffer.length;
    #endif
}

// 응답 데이터 전송 함수
void Modbus_SendResponseData(Modbus_t *modbus) {
    #ifdef DEBUG_MODE
    modbus->status.txStatus = TX_IN_PROGRESS;
    modbus->registers.holdingRegisters[DEBUG_TX_STATUS_ADDR] = modbus->status.txStatus;
    #endif
    #ifdef ASCII_MODE
    // ASCII 모드인 경우 지연 없이 전송
    if (modbus->status.currentParseMode == MODBUS_MODE_ASCII) {
        // 응답 전송
        if (HAL_UART_Transmit_DMA(modbus->huart, modbus->buffers.txFrameBuffer.data, modbus->buffers.txFrameBuffer.length) != HAL_OK) {
            modbus->status.txStatus = TX_ERROR;
            modbus->registers.holdingRegisters[DEBUG_TX_STATUS_ADDR] = modbus->status.txStatus;
            modbus->status.errorCounter++;
            modbus->registers.holdingRegisters[DEBUG_ERROR_COUNT_ADDR] = modbus->status.errorCounter;
            Error_Handler();
        }
    }
    // RTU 모드인 경우 3.5 문자 시간 지연 후 전송
    else {
        uint32_t baudrate = modbus->huart->Init.BaudRate;

        // 2. 지연 시간 계산 (ms 단위 10bit, double로 계산하여 정밀도 유지)
        double delay_ms = 35000.0 / baudrate;

        // 3. 계산된 지연 시간을 정수로 올림하고 최소 2ms 보장
        uint32_t delay_ms_int = (uint32_t)ceil(delay_ms);
        if (delay_ms_int < 2) {
            delay_ms_int = 2;
        }

        // 4. 계산된 지연 시간 적용
        HAL_Delay(delay_ms_int);

        // UART를 통해 응답 전송
        if (HAL_UART_Transmit_DMA(modbus->huart, modbus->buffers.txFrameBuffer.data, modbus->buffers.txFrameBuffer.length) != HAL_OK) {
            #ifdef DEBUG_MODE
            modbus->status.txStatus = TX_ERROR;
            modbus->registers.holdingRegisters[DEBUG_TX_STATUS_ADDR] = modbus->status.txStatus;
            modbus->status.errorCounter++;
            modbus->registers.holdingRegisters[DEBUG_ERROR_COUNT_ADDR] = modbus->status.errorCounter;
            #endif
        }
    }
    #else
        uint32_t baudrate = modbus->huart->Init.BaudRate;

        // 2. 지연 시간 계산 (ms 단위 10bit, double로 계산하여 정밀도 유지)
        double delay_ms = 35000.0 / baudrate;

        // 3. 계산된 지연 시간을 정수로 올림하고 최소 2ms 보장
        uint32_t delay_ms_int = (uint32_t)ceil(delay_ms);
        if (delay_ms_int < 2) {
            delay_ms_int = 2;
        }

        // 4. 계산된 지연 시간 적용
        // HAL_Delay(delay_ms_int); ?

        // UART를 통해 응답 전송
        if (HAL_UART_Transmit_DMA(modbus->huart, modbus->buffers.txFrameBuffer.data, modbus->buffers.txFrameBuffer.length) != HAL_OK) {
            #ifdef DEBUG_MODE
            modbus->status.txStatus = TX_ERROR;
            modbus->registers.holdingRegisters[DEBUG_TX_STATUS_ADDR] = modbus->status.txStatus;
            modbus->status.errorCounter++;
            modbus->registers.holdingRegisters[DEBUG_ERROR_COUNT_ADDR] = modbus->status.errorCounter;
            #endif
            Error_Handler();
        }
    #endif
}

bool push_frame(Modbus_t *modbus, const Frame_t *newFrame) {
#if NUM_RING_BUFFERS <= 2
    // 링버퍼 사용하지 않고 단일 버퍼에 덮어쓰기
    modbus->buffers.rxRingFrameBuffer[0].length = newFrame->length;
    if (newFrame->length > 0) {
        memcpy(modbus->buffers.rxRingFrameBuffer[0].data, newFrame->data, BUFFER_SIZE);
    }
    modbus->buffers.rxWriteIndex = 0;
    modbus->buffers.rxReadIndex = 0;
    return true;
#else
    // 기존 ring buffer 방식
    modbus->buffers.rxRingFrameBuffer[modbus->buffers.rxWriteIndex].length = newFrame->length;
    if (newFrame->length > 0) {
        memcpy(modbus->buffers.rxRingFrameBuffer[modbus->buffers.rxWriteIndex].data, 
               newFrame->data, BUFFER_SIZE);
    }

    uint8_t nextIndex = (modbus->buffers.rxWriteIndex + 1) % NUM_RING_BUFFERS;

    if (nextIndex == modbus->buffers.rxReadIndex) {
        modbus->buffers.rxReadIndex = (modbus->buffers.rxReadIndex + 1) % NUM_RING_BUFFERS;
        modbus->status.bufferStatus = RINGBUF_OVERRUN;
        modbus->status.lastErrorCode = RINGBUF_OVERRUN;
        modbus->status.errorCounter++;
        #ifdef MODBUS_DEBUG
        modbus->registers.holdingRegisters[DEBUG_BUFFER_STATUS_ADDR] = modbus->status.bufferStatus;
        #endif
        modbus->buffers.rxWriteIndex = nextIndex;
        return false;
    }

    modbus->buffers.rxWriteIndex = nextIndex;
    return true;
#endif
}


// 프레임 큐에서 프레임을 가져오는 함수
bool pop_frame(Modbus_t *modbus, Frame_t *outFrame) {
#if NUM_RING_BUFFERS <= 2
    if (!modbus->buffers.frameReadyFlag) return false;

    outFrame->length = modbus->buffers.rxRingFrameBuffer[0].length;
    if (outFrame->length > 0) {
        memcpy(outFrame->data, modbus->buffers.rxRingFrameBuffer[0].data, BUFFER_SIZE);
    }

    if (outFrame->data[0] == 0x3A) {
        modbus->status.currentParseMode = MODBUS_MODE_ASCII;
    } else {
        modbus->status.currentParseMode = MODBUS_MODE_RTU;
    }

    modbus->buffers.frameReadyFlag = false;
    return true;
#else
    if (modbus->buffers.rxWriteIndex == modbus->buffers.rxReadIndex) {
        return false; // 큐가 비어있음
    }
    
    uint16_t validLength = modbus->buffers.rxRingFrameBuffer[modbus->buffers.rxReadIndex].length;
    outFrame->length = validLength;
    
    // 실제 유효한 데이터만 복사합니다.
    if (validLength > 0) {
        memcpy(outFrame->data, 
               modbus->buffers.rxRingFrameBuffer[modbus->buffers.rxReadIndex].data, 
               BUFFER_SIZE);
    }

    // 프레임의 첫 바이트로 모드 감지
    if (outFrame->data[0] == 0x3A) {
        modbus->status.currentParseMode = MODBUS_MODE_ASCII;
    } else {
        modbus->status.currentParseMode = MODBUS_MODE_RTU;
    }

    modbus->buffers.rxReadIndex = (modbus->buffers.rxReadIndex + 1) % NUM_RING_BUFFERS;
    return true;
#endif
}

// 링 버퍼에서 프레임을 파싱하는 함수
static bool parse_frame_from_ring_buffer(Modbus_t *modbus) {
    // 링 버퍼에서 프레임 데이터 가져오기
    if (!pop_frame(modbus, &modbus->buffers.currentFrame)) {
        modbus->status.parseStatus = PARSE_ERROR_LENGTH;
        #ifdef DEBUG_MODE
        modbus->registers.holdingRegisters[DEBUG_PARSE_STATUS_ADDR] = modbus->status.parseStatus;
        modbus->status.lastErrorCode = PARSE_ERROR_LENGTH;
        modbus->registers.holdingRegisters[DEBUG_LAST_ERROR_CODE_ADDR] = modbus->status.lastErrorCode;
        #endif
        return false; // 프레임을 가져오지 못함
    }
    
    // 프레임 파싱
    if (!ModbusRTU_ParseFrame(modbus)) {
        Modbus_ResetRx(modbus); // 또는 해당 프레임 무시 처리
        return false;
    }

    return true;
}

void Modbus_CheckUART(Modbus_t *modbus) {
    static uint32_t lastLength = 0;
    static uint32_t lastCheckTick = 0;

    if (HAL_GetTick() - lastCheckTick >= 100) {
        lastCheckTick = HAL_GetTick();

        if (modbus->buffers.rxFrameBuffer.length == lastLength) {
            if ((modbus->huart->Instance->CR1 & USART_CR1_RXNEIE) == 0) {
                Modbus_ResetRx(modbus);
            }
        }

        lastLength = modbus->buffers.rxFrameBuffer.length;
    }
}

void Modbus_ResetRx(Modbus_t *modbus) {
    // 버퍼 클리어
    memset(modbus->buffers.rxFrameBuffer.data, 0, BUFFER_SIZE);
    modbus->buffers.rxFrameBuffer.length = 0;

    // 상태 초기화
    modbus->buffers.firstDataFlag = true;

    // UART 수신 재시작
    HAL_UART_Abort_IT(modbus->huart);
    HAL_UART_Receive_IT(modbus->huart, modbus->buffers.rxFrameBuffer.data, 1);

    #ifdef DEBUG_MODE
    modbus->status.lastErrorCode = RX_RECOVERY_TRIGGERED;
    modbus->status.rxStatus = RX_RECOVERY_TRIGGERED;
    modbus->status.errorCounter++;
    modbus->registers.holdingRegisters[DEBUG_RX_STATUS_ADDR] = RX_RECOVERY_TRIGGERED;
    #endif
}