// modbus_rtu.c
#include "modbus_rtu.h"
#include <string.h>

// RTU 모드 초기화 함수
void ModbusRTU_Init(Modbus_t *modbus) {
    // RTU 모드 관련 초기화 작업
    modbus->status.currentRxMode = MODBUS_MODE_RTU;
    modbus->status.currentParseMode = MODBUS_MODE_RTU;
}

// RTU 프레임 파싱 함수
bool ModbusRTU_ParseFrame(Modbus_t *modbus) {
    // 프레임 유효성 검사 (최소 길이)
    if (modbus->buffers.currentFrame.length < 8) { // 최소 길이: SlaveID(1) + FunctionCode(1) + Data(4) + CRC(2)
        modbus->status.parseStatus = PARSE_ERROR_LENGTH2;
        #ifdef DEBUG_MODE
        modbus->registers.holdingRegisters[DEBUG_PARSE_STATUS_ADDR] = modbus->status.parseStatus;
        modbus->registers.holdingRegisters[DEBUG_RX_COUNT_ADDR] = modbus->buffers.currentFrame.length; // 실제 길이 저장
        modbus->status.lastErrorCode = PARSE_ERROR_LENGTH2;
        modbus->registers.holdingRegisters[DEBUG_LAST_ERROR_CODE_ADDR] = modbus->status.lastErrorCode;
        modbus->status.errorCounter++;
        modbus->registers.holdingRegisters[DEBUG_ERROR_COUNT_ADDR] = modbus->status.errorCounter;
        #endif
        return false; // 길이 오류
    }
    
    // CRC 확인
    uint16_t receivedCRC = (modbus->buffers.currentFrame.data[modbus->buffers.currentFrame.length - 2]) | 
                          (modbus->buffers.currentFrame.data[modbus->buffers.currentFrame.length - 1] << 8);
    uint16_t calculatedCRC = ModbusRTU_CalculateCRC(modbus->buffers.currentFrame.data, modbus->buffers.currentFrame.length - 2);
    #ifdef DEBUG_MODE
    modbus->registers.holdingRegisters[DEBUG_TX_LENGTH_ADDR] = receivedCRC;
    modbus->registers.holdingRegisters[DEBUG_FRAME_LENGTH_ADDR] = calculatedCRC;
    #endif
    if (receivedCRC != calculatedCRC) {
        modbus->status.parseStatus = RX_ERROR_CRC;
        #ifdef DEBUG_MODE
        modbus->registers.holdingRegisters[DEBUG_PARSE_STATUS_ADDR] = modbus->status.parseStatus;
        modbus->status.lastErrorCode = RX_ERROR_CRC;
        modbus->registers.holdingRegisters[DEBUG_LAST_ERROR_CODE_ADDR] = modbus->status.lastErrorCode;
        modbus->status.errorCounter++;
        modbus->registers.holdingRegisters[DEBUG_ERROR_COUNT_ADDR] = modbus->status.errorCounter;
        #endif
        return false; // CRC 오류
    }
    
    // 유효한 프레임 (디버깅용 정보 저장)
    #ifdef DEBUG_MODE
    modbus->status.parseStatus = PARSE_OK;
    modbus->registers.holdingRegisters[DEBUG_PARSE_STATUS_ADDR] = modbus->status.parseStatus;
    #endif
    return true; // 파싱 성공
}

// RTU 프레임 수신 완료 처리 함수
void ModbusRTU_RxFrameComplete(Modbus_t *modbus, UART_HandleTypeDef *huart) {
    modbus->status.rxStatus = RX_COMPLETE;
    #ifdef DEBUG_MODE
    modbus->registers.holdingRegisters[DEBUG_RX_STATUS_ADDR] = modbus->status.rxStatus;

    modbus->registers.holdingRegisters[DEBUG_RX_LENGTH_ADDR] = modbus->buffers.rxFrameBuffer.length;
    #endif
    // 링 버퍼에 프레임 저장
    Frame_t tempFrame;
    memcpy(&tempFrame, &modbus->buffers.rxFrameBuffer, sizeof(Frame_t));
    
    if (push_frame(modbus, &tempFrame)) {
        // 프레임 프로세스 플래그 ON
        modbus->buffers.frameReadyFlag = true;
        modbus->status.bufferStatus = RINGBUF_OK;
    } 
    #ifdef DEBUG_MODE
    else {
        modbus->status.bufferStatus = RINGBUF_OVERRUN;
        modbus->registers.holdingRegisters[DEBUG_BUFFER_STATUS_ADDR] = modbus->status.bufferStatus;
        modbus->status.errorCounter++;
        modbus->registers.holdingRegisters[DEBUG_ERROR_COUNT_ADDR] = modbus->status.errorCounter;
    }
    
    modbus->status.rxStatus = RX_IDLE;
    modbus->registers.holdingRegisters[DEBUG_RX_STATUS_ADDR] = modbus->status.rxStatus;
    #endif
}

// CRC 계산 함수
uint16_t ModbusRTU_CalculateCRC(uint8_t *data, uint16_t length) {
    uint16_t crc = 0xFFFF;

    for (uint16_t i = 0; i < length; i++) {
        crc ^= data[i];
        for (uint8_t j = 0; j < 8; j++) {
            if (crc & 0x0001) {
                crc = (crc >> 1) ^ 0xA001;
            } else {
                crc >>= 1;
            }
        }
    }

    return crc;
}
