/*
 * Baremetal_Tasks.h
 *
 *  Created on: Dec 1, 2025
 *      Author: c6h6
 */

#ifndef INC_BAREMETAL_TASKS_H_
#define INC_BAREMETAL_TASKS_H_

#ifdef TASKS_AS_FREERTOS
#include "FreeRTOS.h"
#include "task.h"
#include "cmsis_os.h"
#endif

#include <stdint.h>
#include <stdbool.h>
#include "main.h" // STM32 HAL definitions

// =============================================================
// 1. 공용 캐시 (Shared Cache) 및 데이터 구조 정의
// =============================================================

// 센서 인덱스 정의 (ADC 채널 순서에 맞춰 매핑 필요)
typedef enum
{
	// --- TDS & Temperature ---
	SENSOR_TDS_RAW = 0,         // SEN0244 (Analog)
	SENSOR_ELECTROLYTE_TEMP,    // RTD or NTC (For TDS Compensation)

	// --- Current Sensors (Shunt + Amp) ---
	SENSOR_TANK_ADD_PUMP_CUR,
	SENSOR_CARTRIDGE_HEATER_CUR,
	SENSOR_DC_FAN_CUR,
	SENSOR_E_ROTATE_PUMP_CUR,
	SENSOR_H2_VALVE_1_CUR,
	SENSOR_H2_VALVE_2_CUR,
	SENSOR_H2_VALVE_3_CUR,

	// --- Voltage/Flow/Pressure Sensors ---
	SENSOR_O2_FLOW,
	SENSOR_H2_FLOW,
	SENSOR_H2_PRESS_1,
	SENSOR_H2_PRESS_2,

	// --- PSU Monitoring ---
	SENSOR_PSU_VOLTAGE,
	SENSOR_PSU_CURRENT,

	SENSOR_COUNT_TOTAL
} SensorIndex_t;

typedef enum
{
	VALVE_MODE_ATMOS = 0,  // 상압 모드
	VALVE_MODE_HIGHP,      // 고압 모드
	VALVE_MODE_EMERG       // 비상 정지 모드
} valve_mode_t;
// 공용 캐시 구조체
typedef struct
{
	// [System Status]
	volatile bool is_emergency;          // 비상 정지 상태
	volatile bool time_sync_ok;          // 시간 동기화 완료 여부
	volatile uint32_t last_sync_time;    // 마지막 동기화 시간 (Tick)

	// [Control Flags] (Task Supervisor -> Actuators)
	volatile bool psu_enable;            // Task E (Power Supply) 활성화
	volatile bool electrolyte_enable;    // Task B, C, D (전해질 제어) 활성화
	volatile bool high_pressure_mode;    // Task F (고압/상압 모드)

	// [Set Values (SV)] (From HMI via Task H)
	volatile double voltage_sv;
	volatile double current_sv;
	volatile float temp_sv;
	volatile float flow_sv;

	// [Process Values (PV)] (From Task A)
	volatile float tds_ppm;              // 계산된 TDS 값 (ppm)
	volatile float electrolyte_temp;     // 전해질 온도 (도씨)
	volatile float sensor_phys_values[SENSOR_COUNT_TOTAL]; // 변환된 물리량 저장

	// [Digital Inputs]
	volatile bool level_sensor_high;
	volatile bool level_sensor_low;

	// [Actuator Outputs]
	volatile float pump_circulation_duty; // 0.0 ~ 1.0 or Voltage
	volatile bool pump_refill_on;
	volatile bool heater_on;
	volatile bool fan_on;

	// [DO 상태 – 텔레메트리용 echo]
	volatile bool h2sol1_ctrl;          // H2SOLVALVE_1_CTRL (상압 라인)
	volatile bool h2sol2_ctrl;          // H2SOLVALVE_2_CTRL (고압 라인)
	volatile bool h2sol3_ctrl;          // H2SOLVALVE_3_CTRL (벤트 라인)

	volatile bool highp_production_ctrl; // HIGHP_PRODUCTION_CTRL (현재 운전 모드 상태)

	volatile valve_mode_t valve_mode;
	// [Internal PID State]
	float flow_pid_integral;

	// 원본 raw
	volatile int32_t volt_sen_raw, curr_sen_raw, h_gas_p_sen1_raw,
			h_gas_p_sen2_raw;

	// 보정 raw
	volatile int32_t volt_sen_raw_cal, curr_sen_raw_cal, h_gas_p_sen1_raw_cal,
			h_gas_p_sen2_raw_cal;

	// "필터된 보정 raw"  <-- ESP32로 이 값 보내면 됨
	volatile int32_t volt_sen_raw_filt, curr_sen_raw_filt,
			h_gas_p_sen1_raw_filt, h_gas_p_sen2_raw_filt;

	// 물리량(로컬에서 쓰면 편함)
	volatile double volt_sen_mA, h_gas_p_sen1_mA, h_gas_p_sen2_mA;
	volatile double curr_sen_A;

	volatile bool psu48_mode_cv_ctrl;
	volatile bool psu48_mode_cc_ctrl;
} SharedCache_t;

extern volatile SharedCache_t shared_cache;

// =============================================================
// 2. 함수 프로토타입
// =============================================================

// 초기화
void System_Logic_Init(void);

// 메인 루프에서 호출할 단일 진입점
void System_Logic_Loop(void);

// (Optional) 하드웨어 연결을 위해 사용자가 구현해야 할 함수들 (Weak 처리됨)
uint16_t HW_Get_ADC_Value(uint8_t sensor_idx);
void HW_Set_GPIO(uint8_t actuator_id, bool state);
void HW_Set_DAC_Voltage(float voltage);
void HW_UART_Transmit(uint8_t *data, uint16_t len);

void Task_A_Sensors(void);
void Task_B_Refill(void);
void Task_C_Circulation(void);
void Task_D_TempControl(void);
void Task_E_PSU(void);
void Task_F_ApplyValveMode();
void Task_L_UpdateLEDs(void);
void Task_Supervisor(void);

inline double clampf(double x, double lo, double hi)
{
	if (x < lo)
		return lo;
	if (x > hi)
		return hi;
	return x;
}
inline int32_t clamp_code_24s(int32_t x)
{
	// ADS는 signed 24-bit (대략 [-8388608, +8388607]) 범위가 안전
	if (x > 8388607)
		return 8388607;
	if (x < -8388608)
		return -8388608;
	return x;
}
#ifdef TASKS_AS_FREERTOS
inline float ticks_to_sec(TickType_t dt_ticks)
{
	return ((float) dt_ticks * (float) portTICK_PERIOD_MS) / 1000.0f;
}
#endif
#endif /* INC_BAREMETAL_TASKS_H_ */
