#include "adc_mux_app.h"
#include "adc.h"
#include <string.h>

extern ADC_HandleTypeDef hadc1;
extern ADC_HandleTypeDef hadc2;
extern ADC_HandleTypeDef hadc3;

static uint16_t s_adc1_dma[ADC1_NUM_CH];
static uint16_t s_adc2_dma[ADC2_NUM_CH];
static uint16_t s_adc3_dma[ADC3_NUM_CH];

uint16_t s_adc1_shadow[ADC1_NUM_CH];
uint16_t s_adc2_shadow[ADC2_NUM_CH];
uint16_t s_adc3_shadow[ADC3_NUM_CH];

volatile uint8_t g_adc1_new = 0, g_adc2_new = 0, g_adc3_new = 0;

/* ===================== 필터 설정 ===================== */
/*
 * IIR_K가 클수록 더 부드럽게(느리게) 따라감.
 * - 4  : 빠른 반응, 약간만 완화
 * - 8  : 보통
 * - 16 : 많이 안정화(추천 시작점)
 * - 32 : 매우 안정화(대신 응답 느림)
 */
#define ADC_IIR_K   (16U)     // 2^n일 필요는 없음 (정수 나눗셈)
#define ADC_Q       (16U)     // Q16.16 고정소수점

static uint32_t s_adc1_filt_q[ADC1_NUM_CH];
static uint32_t s_adc2_filt_q[ADC2_NUM_CH];
static uint32_t s_adc3_filt_q[ADC3_NUM_CH];

static uint8_t s_adc1_filt_init = 0;
static uint8_t s_adc2_filt_init = 0;
static uint8_t s_adc3_filt_init = 0;
/* ==================================================== */

const uint16_t* ADC1_Data(void)
{
	return s_adc1_shadow;
}
const uint16_t* ADC2_Data(void)
{
	return s_adc2_shadow;
}
const uint16_t* ADC3_Data(void)
{
	return s_adc3_shadow;
}
int ADC1_Len(void)
{
	return ADC1_NUM_CH;
}
int ADC2_Len(void)
{
	return ADC2_NUM_CH;
}
int ADC3_Len(void)
{
	return ADC3_NUM_CH;
}

// 연속 변환 + DMA(원형) 시작
void ADC_APP_Start(void)
{
	// 필터 상태 리셋(재시작 시 첫 샘플로 초기화되게)
	s_adc1_filt_init = 0;
	s_adc2_filt_init = 0;
	s_adc3_filt_init = 0;

	HAL_ADC_Start_DMA(&hadc1, (uint32_t*) s_adc1_dma, ADC1_NUM_CH);
	HAL_ADC_Start_DMA(&hadc2, (uint32_t*) s_adc2_dma, ADC2_NUM_CH);
	HAL_ADC_Start_DMA(&hadc3, (uint32_t*) s_adc3_dma, ADC3_NUM_CH);
}

static inline uint16_t iir_update_u16(uint32_t *state_q, uint16_t x_u16)
{
	uint32_t x_q = ((uint32_t) x_u16) << ADC_Q;

	// y = y + (x - y)/K  (정수 IIR)
	int32_t diff = (int32_t) x_q - (int32_t) (*state_q);
	*state_q = (uint32_t) ((int32_t) (*state_q) + (diff / (int32_t) ADC_IIR_K));

	// 반올림 후 u16로
	return (uint16_t) ((*state_q + (1U << (ADC_Q - 1U))) >> ADC_Q);
}

// DMA 완료시 “필터 적용된 shadow” 갱신 & 플래그
static inline void snapshot_and_flag(ADC_HandleTypeDef *hadc)
{
	if (hadc == &hadc1)
	{
		if (!s_adc1_filt_init)
		{
			for (int i = 0; i < ADC1_NUM_CH; i++)
			{
				s_adc1_filt_q[i] = ((uint32_t) s_adc1_dma[i]) << ADC_Q;
				s_adc1_shadow[i] = s_adc1_dma[i];
			}
			s_adc1_filt_init = 1;
		}
		else
		{
			for (int i = 0; i < ADC1_NUM_CH; i++)
			{
				s_adc1_shadow[i] = iir_update_u16(&s_adc1_filt_q[i],
						s_adc1_dma[i]);
			}
		}
		g_adc1_new = 1;
	}
	else if (hadc == &hadc2)
	{
		if (!s_adc2_filt_init)
		{
			for (int i = 0; i < ADC2_NUM_CH; i++)
			{
				s_adc2_filt_q[i] = ((uint32_t) s_adc2_dma[i]) << ADC_Q;
				s_adc2_shadow[i] = s_adc2_dma[i];
			}
			s_adc2_filt_init = 1;
		}
		else
		{
			for (int i = 0; i < ADC2_NUM_CH; i++)
			{
				s_adc2_shadow[i] = iir_update_u16(&s_adc2_filt_q[i],
						s_adc2_dma[i]);
			}
		}
		g_adc2_new = 1;
	}
	else if (hadc == &hadc3)
	{
		if (!s_adc3_filt_init)
		{
			for (int i = 0; i < ADC3_NUM_CH; i++)
			{
				s_adc3_filt_q[i] = ((uint32_t) s_adc3_dma[i]) << ADC_Q;
				s_adc3_shadow[i] = s_adc3_dma[i];
			}
			s_adc3_filt_init = 1;
		}
		else
		{
			for (int i = 0; i < ADC3_NUM_CH; i++)
			{
				s_adc3_shadow[i] = iir_update_u16(&s_adc3_filt_q[i],
						s_adc3_dma[i]);
			}
		}
		g_adc3_new = 1;
	}
}

// HAL 콜백 오버라이드
void HAL_ADC_ConvCpltCallback(ADC_HandleTypeDef *hadc)
{
	snapshot_and_flag(hadc);
}

void HAL_ADC_ConvHalfCpltCallback(ADC_HandleTypeDef *hadc)
{
	/* 필요하면 여기서도 snapshot_and_flag(hadc) 호출 가능
	 (갱신 주기 2배로 늘려 필터 반응 개선 가능) */
}

void ADC_APP_Poll(void)
{
	if (g_adc1_new)
	{
		g_adc1_new = 0;
	}
	if (g_adc2_new)
	{
		g_adc2_new = 0;
	}
	if (g_adc3_new)
	{
		g_adc3_new = 0;
	}
}
