/* Ethernet Basic Example

   This example code is in the Public Domain (or CC0 licensed, at your option.)

   Unless required by applicable law or agreed to in writing, this
   software is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
   CONDITIONS OF ANY KIND, either express or implied.
*/
#include "esp_eth.h"
#include "esp_eth_driver.h"
#include "esp_event.h"
#include "esp_log.h"
#include "esp_netif.h"
#include "ethernet_init.h"
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "hal/eth_types.h"
#include "hal/gpio_types.h"
#include "hal/uart_types.h"
#include "lwip/ip4_addr.h"
#include "portmacro.h"
#include "sdkconfig.h"
#include <driver/gpio.h>
#include <driver/uart.h>
#include <math.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
// hx2k_telemetry.c
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
// #include "hx2k_telemetry.h"
#include "esp_log.h"
#include <dummy_telemetry.h>

#include "mqtt_client.h" // esp-idf mqtt

static const char *TAG = "eth_example";
const char *topic = "hx2k/telemetry";
static const char *HX2K_DEVICE_ID_STR = "HXB2k";
static const char *HX2K_SCHEMA_STR = "hxb2k-icd:2.0";

char dev_uuid[5] = "ZZZZ";
bool eth_get = false;
extern esp_mqtt_client_handle_t g_mqtt_client;

#define TXD0_PIN (GPIO_NUM_16)
#define RXD0_PIN (GPIO_NUM_17)

static const int RX_BUF_SIZE = 256;

static void hxb2_tlm_gpio_toggle(void) {
	static uint8_t s_level = 0;

	s_level = !s_level;
	gpio_set_level(GPIO_NUM_2, s_level);
}

extern hxb_desired_shadow_t s_desired_shadow;

typedef struct {
	uint32_t seq;
	uint32_t ms;

	// NEW: STM32 RTC에서 넘어온 시간
	char uuid[5];
	bool rtc_valid;
	uint16_t rtc_year;
	uint8_t rtc_month;
	uint8_t rtc_day;
	uint8_t rtc_hour;
	uint8_t rtc_minute;
	uint8_t rtc_second;

	int16_t t1_x10;
	int16_t t2_x10;
	int16_t t3_x10;

	uint16_t adc1[8];
	uint16_t adc2[8];
	uint16_t adc3[3];
	int32_t sdadc[4];

	uint32_t fan_rpm;
	uint32_t pump_hz;
	uint32_t fan_duty_percent;
	uint32_t pump_dac_mV;
	uint32_t circ_dac_mV;

	uint8_t di[13];

} hxb2_telemetry_t;

static long parse_long(const char *s, long def) {
	if (s == NULL || *s == '\0')
		return def;
	char *end = NULL;
	long v = strtol(s, &end, 10);
	if (end == s)
		return def;
	return v;
}

static bool parse_stm32_tlm_line(const char *line, hxb2_telemetry_t *out) {
	if (!line || !out)
		return false;

	// 헤더 확인
	if (strncmp(line, "HX2K,T,", 7) != 0) {
		return false;
	}

	// 작업용 포인터
	const char *p = line + 7; // "HX2K,T," 뒤에서 시작

	// seq, ms, date, time, t1, t2, t3 를 먼저 sscanf 로 빼고
	uint32_t seq, ms;
	int year, month, day, hour, minute, second;
	int t1_i, t2_i, t3_i;
	char uuid[5];

	int n = sscanf(p, "%[^,],%lu,%lu,%d-%d-%d,%d:%d:%d,%d,%d,%d", uuid, &seq,
				   &ms, &year, &month, &day, &hour, &minute, &second, &t1_i,
				   &t2_i, &t3_i);

	if (n != 12) {
		// 기대한 개수만큼 못 읽었으면 실패
		return false;
	}

	memset(out, 0, sizeof(*out));
	strcpy(dev_uuid, uuid);
	strcpy(out->uuid, uuid);
	ESP_LOGI("UID", "%s", dev_uuid);
	out->seq = seq;
	out->ms = ms;

	// RTC 정보 저장
	out->rtc_valid = true;
	out->rtc_year = (uint16_t)year;
	out->rtc_month = (uint8_t)month;
	out->rtc_day = (uint8_t)day;
	out->rtc_hour = (uint8_t)hour;
	out->rtc_minute = (uint8_t)minute;
	out->rtc_second = (uint8_t)second;

	out->t1_x10 = (int16_t)t1_i;
	out->t2_x10 = (int16_t)t2_i;
	out->t3_x10 = (int16_t)t3_i;

	// 이제 p 를 date/time/t1..t3 뒤로 넘긴 다음 나머지 CSV(ADC, RPM, DI)를 기존
	// 로직대로 돌리면 됨. (위에서 사용한 만큼 스킵) 안전하게 하기 위해, 첫 11개
	// 항목을 따로 토큰화하고, 나머지는 strtok 로 돌려도 되고,
	//  간단히는 첫 번째 쉼표를 11번 건너뛰는 방법도 있음.

	// --- 간단 구현: 쉼표 11번 건너뛰고 나머지를 파싱 ---
	p = line; // 처음으로 리셋
	int comma_cnt = 0;
	while (*p && comma_cnt < 10) {
		if (*p == ',')
			comma_cnt++;
		p++;
	}
	// 이제 p 는 ADC1[0] 이 시작되는 위치 근처일 것
	// 형식:
	// <adc1_0>,<adc1_1>,...,<adc1_N>,<adc2_0>,...,<adc3_N>,rpm,hz,duty,pump_dac,circ_dac,DI...
	// ESP_LOGI(TAG, "UART line: %s", p);
	// ADC1
	for (int i = 0; i < 8; i++) {
		if (*p == '\0')
			return true; // 데이터가 덜 온 경우 그냥 조용히 리턴
		out->adc1[i] = (uint16_t)strtoul(p, (char **)&p, 10);
		if (*p == ',')
			p++;
	}

	// ADC2
	for (int i = 0; i < 8; i++) {
		if (*p == '\0')
			return true;
		out->adc2[i] = (uint16_t)strtoul(p, (char **)&p, 10);
		if (*p == ',')
			p++;
	}

	// ADC3
	for (int i = 0; i < 3; i++) {
		if (*p == '\0')
			return true;
		out->adc3[i] = (uint16_t)strtoul(p, (char **)&p, 10);
		if (*p == ',')
			p++;
	}
	// sigma-delta ADC
	for (int i = 0; i < 4; i++) {
		if (*p == '\0')
			return true;
		out->sdadc[i] = (int32_t)strtoul(p, (char **)&p, 10);
		if (*p == ',')
			p++;
	}

	// RPM, HZ, duty, pump_dac, circ_dac
	if (*p == '\0')
		return true;
	out->fan_rpm = strtoul(p, (char **)&p, 10);
	if (*p == ',')
		p++;

	if (*p == '\0')
		return true;
	out->pump_hz = strtoul(p, (char **)&p, 10);
	if (*p == ',')
		p++;

	if (*p == '\0')
		return true;
	out->fan_duty_percent = strtoul(p, (char **)&p, 10);
	if (*p == ',')
		p++;

	if (*p == '\0')
		return true;
	out->pump_dac_mV = strtoul(p, (char **)&p, 10);
	if (*p == ',')
		p++;

	if (*p == '\0')
		return true;
	out->circ_dac_mV = strtoul(p, (char **)&p, 10);
	if (*p == ',')
		p++;

	// DI 13개
	for (int i = 0; i < 13; i++) {
		if (*p == '\0')
			break;
		out->di[i] = (uint8_t)strtoul(p, (char **)&p, 10);
		if (*p == ',')
			p++;
	}

	return true;
}

static void build_iso8601_utc(char *buf, size_t buf_len) {
	// SNTP로 시간 동기화 되어있다는 가정.
	time_t now = 0;
	time(&now);

	struct tm tm;
	gmtime_r(&now, &tm);

	// 예: "2025-10-18T16:00:00Z"
	strftime(buf, buf_len, "%Y-%m-%dT%H:%M:%SZ", &tm);
}
char json[2048];		   // AI 32개 + DI 17개 + 헤더 여유
#define FS_23 (8388607.0f) // 2^23 - 1
#define VREF_ADC (1.2f)
#define R_SHUNT_4_20MA (51.0f)
#define ADC_VOLTAGE_OFFSET_COMP (0.0f)
float adc_to_mA(uint16_t adc_raw) {
	const float VREF = 3.3f;
	const float R_SHUNT = 150.0f; // R159

	float v_adc = (adc_raw * VREF) / 4095.0f;
	float i_mA = (v_adc / R_SHUNT) * 1000.0f;

	return i_mA;
}
// ============================================================
// 전압 센서 RAW -> 이상적 RAW 변환 2-Point 캘리브레이션 계수
// ============================================================
// 1. 캘리브레이션 데이터 구조체 정의
typedef struct {
	int32_t raw_code;
	double voltage;
} calib_point_t;

// 2. 실측 데이터 테이블 (반드시 RAW 코드 기준 오름차순으로 적어야 함!)
static const calib_point_t volt_calib_table[] = {
	{1485209, 0.00f},  {4861643, 32.00f}, {5190003, 35.00f},
	{5736422, 40.03f}, {6159454, 43.8f},  {6420777, 46.2f}
};

// 테이블의 데이터 개수 자동 계산
#define VOLT_CALIB_PTS (sizeof(volt_calib_table) / sizeof(volt_calib_table[0]))

// 3. 구간별 선형 보간 함수
static inline double code_to_Voltage_piecewise(int32_t raw_filt) {
	// Case A: 가장 낮은 구간(0V)보다 값이 작게 들어올 때 (외삽법)
	if (raw_filt <= volt_calib_table[0].raw_code) {
		double dx = (double)(volt_calib_table[1].raw_code -
						   volt_calib_table[0].raw_code);
		double dy = volt_calib_table[1].voltage - volt_calib_table[0].voltage;
		return volt_calib_table[0].voltage +
			   ((double)(raw_filt - volt_calib_table[0].raw_code) * (dy / dx));
	}

	// Case B: 가장 높은 구간(41.68V)보다 값이 크게 들어올 때 (외삽법)
	if (raw_filt >= volt_calib_table[VOLT_CALIB_PTS - 1].raw_code) {
		int last = VOLT_CALIB_PTS - 1;
		double dx = (double)(volt_calib_table[last].raw_code -
						   volt_calib_table[last - 1].raw_code);
		double dy =
			volt_calib_table[last].voltage - volt_calib_table[last - 1].voltage;
		return volt_calib_table[last].voltage +
			   ((double)(raw_filt - volt_calib_table[last].raw_code) *
				(dy / dx));
	}

	// Case C: 정상적인 구간 내에 있을 때 (구간 탐색 후 보간)
	for (int i = 0; i < VOLT_CALIB_PTS - 1; i++) {
		if (raw_filt >= volt_calib_table[i].raw_code &&
			raw_filt <= volt_calib_table[i + 1].raw_code) {
			float dx = (float)(volt_calib_table[i + 1].raw_code -
							   volt_calib_table[i].raw_code);
			float dy =
				volt_calib_table[i + 1].voltage - volt_calib_table[i].voltage;
			return volt_calib_table[i].voltage +
				   ((float)(raw_filt - volt_calib_table[i].raw_code) *
					(dy / dx));
		}
	}

	return 0.0f; // Fallback (이론상 도달하지 않음)
}


// ============================================================
// 압력 센서 RAW -> 이상적 RAW 변환 2-Point 캘리브레이션 계수
// ============================================================
// 1. 캘리브레이션 데이터 구조체 정의
typedef struct {
	int32_t raw_code;
	double atm;
} atm_calib_point_t;

// 2. 실측 데이터 테이블 (반드시 RAW 코드 기준 오름차순으로 적어야 함!)
static const atm_calib_point_t atm_calib_table[] = {
	{1483383, 0.571f},  {1554588, 1.211f}, {1621493, 1.8f},
	{1798908, 3.433f}, {1992403, 5.174f},  {2133239, 6.458f},{2215000, 7.187f}
};

// 테이블의 데이터 개수 자동 계산
#define ATM_CALIB_PTS (sizeof(atm_calib_table) / sizeof(atm_calib_table[0]))

// 3. 구간별 선형 보간 함수
static inline double code_to_ATM_piecewise(int32_t raw_filt) {
	// Case A: 가장 낮은 구간(0V)보다 값이 작게 들어올 때 (외삽법)
	if (raw_filt <= atm_calib_table[0].raw_code) {
		double dx = (double)(atm_calib_table[1].raw_code -
						   atm_calib_table[0].raw_code);
		double dy = atm_calib_table[1].atm - atm_calib_table[0].atm;
		return atm_calib_table[0].atm +
			   ((double)(raw_filt - atm_calib_table[0].raw_code) * (dy / dx));
	}

	// Case B: 가장 높은 구간(41.68V)보다 값이 크게 들어올 때 (외삽법)
	if (raw_filt >= atm_calib_table[ATM_CALIB_PTS - 1].raw_code) {
		int last = ATM_CALIB_PTS - 1;
		double dx = (double)(atm_calib_table[last].raw_code -
						   atm_calib_table[last - 1].raw_code);
		double dy =
			atm_calib_table[last].atm - atm_calib_table[last - 1].atm;
		return atm_calib_table[last].atm +
			   ((double)(raw_filt - atm_calib_table[last].raw_code) *
				(dy / dx));
	}

	// Case C: 정상적인 구간 내에 있을 때 (구간 탐색 후 보간)
	for (int i = 0; i < ATM_CALIB_PTS - 1; i++) {
		if (raw_filt >= atm_calib_table[i].raw_code &&
			raw_filt <= atm_calib_table[i + 1].raw_code) {
			double dx = (float)(atm_calib_table[i + 1].raw_code -
							   atm_calib_table[i].raw_code);
			double dy =
				atm_calib_table[i + 1].atm - atm_calib_table[i].atm;
			return atm_calib_table[i].atm +
				   ((float)(raw_filt - atm_calib_table[i].raw_code) *
					(dy / dx));
		}
	}

	return 0.0f; // Fallback (이론상 도달하지 않음)
}

static inline double code_to_mA_4_20(int32_t code_filt) {
	double v = ((double)code_filt / FS_23) * VREF_ADC;
	return (v / R_SHUNT_4_20MA) * 1000.0f;
}

// HASS-50S + INA Gain=1(Rg 미삽): I(A)=code/FS * 96
static inline double code_to_A_hass50s(int32_t code_filt) {
	return ((double)code_filt / FS_23) * 96.0f;
}

float Solenoid_Current_FromADC(uint16_t adc_raw) {
	const float VREF = 3.3f;
	const float ADC_MAX = 4095.0f;
	const float INA_GAIN = 200.0f;
	const float R_SHUNT = 0.01f; // ohm

	// ADC 코드 → INA 출력 전압
	float v_out = (adc_raw * VREF) / ADC_MAX;

	// INA 출력 전압 → 션트 전류(= 솔레노이드 전류 + LED 전류)
	float i_total = v_out / (INA_GAIN * R_SHUNT); // [A]

	// ----- 옵션: LED 가지 전류 보정 -----
	// 대략 (24V - Vf) / 4.7k. 24V, Vf≈2V면 약 4.7 mA.
	// 실측해서 맞추는 게 좋음.
	const float I_LED = 0.0047f; // [A] 필요 없으면 0.0f 로 두기

	float i_solenoid = i_total - I_LED;
	if (i_solenoid < 0.0f)
		i_solenoid = 0.0f; // 언더플로 보호

	return i_solenoid;
}
float Pump_Current_FromADC(uint16_t adc_raw) {
	const float VREF = 3.3f;
	const float ADC_MAX = 4095.0f;
	const float INA_GAIN = 200.0f;
	const float R_SHUNT = 0.001f; // ohm

	// ADC 코드 → INA 출력 전압
	float v_out = (adc_raw * VREF) / ADC_MAX;

	// INA 출력 전압 → 션트 전류(= 솔레노이드 전류 + LED 전류)
	float i_total = v_out / (INA_GAIN * R_SHUNT); // [A]

	// ----- 옵션: LED 가지 전류 보정 -----
	// 대략 (24V - Vf) / 4.7k. 24V, Vf≈2V면 약 4.7 mA.
	// 실측해서 맞추는 게 좋음.
	const float I_LED = 0; // [A] 필요 없으면 0.0f 로 두기

	float i_solenoid = i_total - I_LED;
	if (i_solenoid < 0.0f)
		i_solenoid = 0.0f; // 언더플로 보호

	return i_solenoid;
}
float Heater_Current_FromADC(uint16_t adc_raw) {
	const float VREF = 2.7f;
	const float ADC_MAX = 4095.0f;
	const float INA_GAIN = 200.0f;
	const float R_SHUNT = 0.001f; // ohm

	// ADC 코드 → INA 출력 전압
	float v_out = (adc_raw * VREF) / ADC_MAX;

	// INA 출력 전압 → 션트 전류(= 솔레노이드 전류 + LED 전류)
	float i_total = v_out / (INA_GAIN * R_SHUNT); // [A]

	// ----- 옵션: LED 가지 전류 보정 -----
	// 대략 (24V - Vf) / 4.7k. 24V, Vf≈2V면 약 4.7 mA.
	// 실측해서 맞추는 게 좋음.
	const float I_LED = 0.0047f; // [A] 필요 없으면 0.0f 로 두기

	float i_solenoid = i_total - I_LED;
	if (i_solenoid < 0.0f)
		i_solenoid = 0.0f; // 언더플로 보호

	return i_solenoid;
}
#define RL_VALUE (1) // define the load resistance on the board, in kilo ohms
#define RO_CLEAN_AIR_FACTOR                                                    \
	(9.21) // RO_CLEAR_AIR_FACTOR=(Sensor resistance in clean air)/RO,
		   // which is derived from the chart in datasheet

/**********************Application Related
 * Macros**********************************/
#define GAS_H2 (0)

/*****************************Globals***********************************************/
float H2Curve[3] = {2.3, 0.93,
					-1.44}; // two points are taken from the curve in datasheet.
							// with these two points, a line is formed which is
							// "approximately equivalent" to the original curve.
							// data format:{ x, y, slope}; point1: (lg200,
							// lg8.5), point2: (lg10000, lg0.03)

float Ro = 1; // Ro is initialized to 10 kilo ohms
float MQResistanceCalculation(uint16_t raw_adc) {
	return (((float)RL_VALUE * (4095 - raw_adc) / raw_adc));
}
int MQGetPercentage(float rs_ro_ratio, float *pcurve) {
	return (
		pow(10, (((log(rs_ro_ratio) - pcurve[1]) / pcurve[2]) + pcurve[0])));
}

int MQGetGasPercentage(float rs_ro_ratio, int gas_id) {
	if (gas_id == GAS_H2) {
		return MQGetPercentage(rs_ro_ratio, H2Curve);
	}
	return 0;
}

#include <stdint.h>

static uint16_t g_zero_raw = 1; // 네가 측정한 0A raw를 일단 기본값으로
static uint8_t g_zero_ok = 1;

// 0A 상태에서 여러 샘플 평균으로 g_zero_raw를 갱신(추천)
void Current_SetZeroRaw(uint16_t zero_raw) {
	g_zero_raw = zero_raw;
	g_zero_ok = 1;
}

// raw -> 전류[A] (0점 보정)
float Current_A_FromADC(uint16_t raw) {
	const float ADC_VREF = 3.3f;
	const float ADC_MAX = 4095.0f;

	// INA128: G = 1 + 50k / RG
	const float INA_RG_OHMS = 51000.0f;
	const float INA_GAIN = (1.0f + (50000.0f / INA_RG_OHMS)); // ≈ 1.980392

	// HASS 50-S: Vout = Uref ± (0.625 * I / IPN), IPN=50A -> 0.0125 V/A
	const float HASS_V_PER_A = (0.625f / 50.0f); // 0.0125
	// if (!g_zero_ok) return 0.0f;

	// Δraw -> ΔVina
	float dv = ((float)((int32_t)raw - (int32_t)1)) * (ADC_VREF / ADC_MAX);

	// I = ΔVina / (G * (V/A))
	float I = dv / (INA_GAIN * HASS_V_PER_A);

	// 부호가 반대로 나오면 아래 한 줄로 뒤집어
	// I = -I;

	return I;
}

enum __DI_FIELDS {
	DIN_TANK1 = 0,
	DIN_TANK2,
	DIN_TANK3,
	DIN_TANK4,
	DIN_CONTACT,
	DIN_SWITCH,
	DIN_DAC_RDY,
	DIN_CONTACTCTL,
	DIN_H_GAS_SOL1_OUT,
	DIN_H_GAS_SOL2_OUT,
	DIN_H_GAS_SOL3_OUT,
	DIN_PSUENCTL,
	DIN_CARTRIDGE_HEATER
};
extern void mqtt_device_topic_subscribe();
void hx2k_publish_telemetry(esp_mqtt_client_handle_t client,
							const hxb2_telemetry_t *f) {
	if (!client || !f) {
		return;
	}

	// 타임스탬프 & 버퍼
	char ts[32];

	{
		// TIME,YYYY-MM-DD,HH:MM:SS 는 기본적으로 UTC라고 가정하고 Z 붙임
		snprintf(ts, sizeof(ts), "%04u-%02u-%02uT%02u:%02u:%02uZ",
				 (unsigned)f->rtc_year, (unsigned)f->rtc_month,
				 (unsigned)f->rtc_day, (unsigned)f->rtc_hour,
				 (unsigned)f->rtc_minute, (unsigned)f->rtc_second);
	} /* else {
		 // 혹시 RTC가 없는 경우에는 ms 기반으로 대충 만든다거나, 빈 문자열로 둘
	 수도 있음 snprintf(ts, sizeof(ts), "1970-01-01T00:00:00Z");
	 }*/

	// ---------- AI 값 준비 (ICD v2.1 기준 전체) ----------
	// 인덱스 대응:
	//  0: TANK_T_DEGC
	//  1: BEFORE_STACK_T_DEGC
	//  2: AFTER_STACK_T_DEGC
	//  3: TOTAL_DISSOLVED_PPM
	//  4: MEASURED_CURRENT_A
	//  5: STACK_VOLTAGE_V
	//  6: H2_LEAK_PPM
	//  7: ELECTROLYTE_LEAK_SW_A
	//  8: O2_FLOW_SLM
	//  9: H2_FLOW_SLM
	// 10: ELEC_FLOW_SLM
	// 11: H2_PRESS1_BAR
	// 12: H2_PRESS2_BAR
	// 13: H2_IN_O2_VOLPCT
	// 14: O2_IN_H2_PPM
	// 15: DEWPOINT1_DEGC
	// 16: DEWPOINT2_DEGC
	// 17: PWMFAN_DUTY_PCT
	// 18: PWMFAN_PULSE_HZ
	// 19: AXIALFAN_STAT_A
	// 20: MAKEUP_PUMP_CONTROL_V
	// 21: MAKEUP_PUMP_STAT_A
	// 22: CIRC_PUMP_CONTROL_V
	// 23: CIRC_PUMP_STAT_A
	// 24: HEATER_STAT_A
	// 25: H2SOLVALVE_1_STAT_A
	// 26: H2SOLVALVE_2_STAT_A
	// 27: H2SOLVALVE_3_STAT_A
	// 28: PSU48_VOLTAGE_SV_V
	// 29: PSU48_CURRENT_SV_A
	// 30: ELECTROLYTE_TEMP_SV_DEGC
	// 31: ELECTROLYTE_FLOW_SV_SLM

	double ai[32] = {0.0};

	// 온도 3개는 STM32 프레임에서 제대로 반영
	ai[0] = (double)f->t1_x10 / 10.0; // TANK_T_DEGC
	ai[1] = (double)f->t2_x10 / 10.0; // BEFORE_STACK_T_DEGC
	ai[2] = (double)f->t3_x10 / 10.0; // AFTER_STACK_T_DEGC

	ai[4] = (code_to_A_hass50s(f->sdadc[1]) >= 3.0)
				? code_to_A_hass50s(f->sdadc[1])
				: 0;
	ai[5] = (code_to_Voltage_piecewise(f->sdadc[0])-ADC_VOLTAGE_OFFSET_COMP) <= 0
				? 0
				: (code_to_Voltage_piecewise(f->sdadc[0])-ADC_VOLTAGE_OFFSET_COMP);

	ai[6] =
		MQGetGasPercentage(MQResistanceCalculation(f->adc1[7]) / Ro, GAS_H2);
	ai[7] = ((f->adc1[6]) * 3.3 / 4095 * 2);
	// ai[8] =

	ai[10] = ((adc_to_mA(f->adc1[0]) - 4.00) * 9.5 / 16);

	ai[11] = (code_to_ATM_piecewise(f->sdadc[2])*16/50) <= 0
				 ? 0
				 : (code_to_ATM_piecewise(f->sdadc[2])*16/50);
	ai[12] = (code_to_ATM_piecewise(f->sdadc[3])*16/50) <= 0
				 ? 0
				 : (code_to_ATM_piecewise(f->sdadc[3])*16/50);
	// 팬 RPM → 펄스 Hz (대충 1 pulse/rev 가정)
	ai[17] = f->fan_duty_percent;
	ai[18] = (f->fan_rpm); // PWMFAN_PULSE_HZ
	ai[19] = Solenoid_Current_FromADC(f->adc1[4]);
	ai[20] = ((f->pump_dac_mV) * 5.0 / 65535);
	ai[21] = Pump_Current_FromADC(f->adc2[3]);

	ai[22] = ((f->circ_dac_mV) * 5.0 / 65535);
	ai[23] = Pump_Current_FromADC(f->adc1[3]);

	ai[24] = Heater_Current_FromADC(f->adc3[0]);
	// 나머지 ai[x] = 0.0;  // 이미 초기화됨
	ai[25] = Solenoid_Current_FromADC(f->adc1[2]);
	ai[26] = Solenoid_Current_FromADC(f->adc3[2]);
	ai[27] = Solenoid_Current_FromADC(f->adc3[1]);
	ai[28] = s_desired_shadow.psu48_v_set;
	ai[29] = s_desired_shadow.psu48_i_set;
	ai[30] = s_desired_shadow.electrolyte_t_sp_degC;
	ai[31] = s_desired_shadow.electrolyte_flow_sp_slm;
	// ---------- DI 값 준비 (ICD Telemetry 예제 + 표 기준 전체) ----------
	// 순서:
	//  0: TANK_PRESS_SW
	//  1: LEVEL_SINGLE
	//  2: LEVEL_DOUBLE_HIGHER
	//  3: LEVEL_DOUBLE_LOWER
	//  4: HEATER_CTRL
	//  5: HEATER_STAT
	//  6: H2SOLVALVE_1_CTRL
	//  7: H2SOLVALVE_2_CTRL
	//  8: H2SOLVALVE_3_CTRL
	//  9: PSU48_MC_CTRL
	// 10: PSU48_MC_STAT
	// 11: PSU48_PWR_ENA_CTRL
	// 12: PSU48_MODE_CV_CTRL
	// 13: PSU48_MODE_CC_CTRL
	// 14: ELECTROLYTE_CTRL_CTRL
	// 15: HIGHP_PRODUCTION_CTRL
	// 16: EMERGENCY_ACTIVE

	bool di_val[17] = {0};

	// STM32 DI 비트와의 매핑 (대부분은 TODO, 확실한 것만 먼저 연결)
	di_val[0] =
		(f->di[DIN_TANK4] != 0); // TANK_PRESS_SW        (탱크 압력 스위치)
	di_val[1] =
		(f->di[DIN_TANK3] != 0); // LEVEL_SINGLE         (싱글 레벨 센서)
	di_val[2] =
		(f->di[DIN_TANK2] != 0); // LEVEL_DOUBLE_HIGHER  (더블 레벨 상부)
	di_val[3] =
		(f->di[DIN_TANK1] != 0); // LEVEL_DOUBLE_LOWER   (더블 레벨 하부)

	di_val[4] = (f->di[DIN_CARTRIDGE_HEATER] != 0);
	di_val[6] = (f->di[DIN_H_GAS_SOL1_OUT] != 0);
	di_val[7] = (f->di[DIN_H_GAS_SOL2_OUT] != 0);
	di_val[8] = (f->di[DIN_H_GAS_SOL3_OUT] != 0);
	// PSU 마그네틱 컨택터 상태 정도는 contact 비트와 묶어둠 (필요시 수정)
	di_val[9] = (f->di[DIN_CONTACTCTL] != 0);
	di_val[10] = (f->di[DIN_CONTACT] != 0); // PSU48_MC_STAT
	di_val[11] = (s_desired_shadow.psu48_ena);
	// 비상정지
	di_val[14] = s_desired_shadow.electrolyte_ctrl_ena;
	di_val[15] = s_desired_shadow.highp_production_ena;
	di_val[16] = (f->di[DIN_SWITCH] != 0); // EMERGENCY_ACTIVE

	// 나머지 DO/펌웨어 변수 기반 DI값들은 일단 false
	// (HEATER_CTRL, HEATER_STAT, H2SOLVALVE_x_CTRL, PSU48_*_CTRL 등)
	// → STM32/펌웨어에서 실제 제어 변수랑 매핑되면 여기 채워주면 됨.

	const char *di_str[17];
	for (int i = 0; i < 17; ++i) {
		di_str[i] = di_val[i] ? "true" : "false";
	}

	// ---------- JSON 한 방에 빌드 (snprintf 1회) ----------
	int len = sprintf(
		json,
		"{"
		"\"schema\":\"%s\","
		"\"device_id\":\"%s-%s\","
		"\"ts\":\"%s\","
		"\"seq_tlm\":%lu,"
		"\"applied_rev\":%lu,"
		"\"telemetry\":{\"AI\":{"
		"\"TANK_T_DEGC\":%.2f,"
		"\"BEFORE_STACK_T_DEGC\":%.2f,"
		"\"AFTER_STACK_T_DEGC\":%.2f,"
		"\"TOTAL_DISSOLVED_PPM\":%.2f,"
		"\"MEASURED_CURRENT_A\":%.2f,"
		"\"STACK_VOLTAGE_V\":%.2f,"
		"\"H2_LEAK_PPM\":%.2f,"
		"\"ELECTROLYTE_LEAK_SW_A\":%.2f,"
		"\"O2_FLOW_SLM\":%.2f,"
		"\"H2_FLOW_SLM\":%.2f,"
		"\"ELECTROLYTE_FLOW_SLM\":%.2f,"
		"\"H2_PRESS1_BAR\":%.2f,"
		"\"H2_PRESS2_BAR\":%.2f,"
		"\"H2_IN_O2_VOLPCT\":%.2f,"
		"\"O2_IN_H2_PPM\":%.2f,"
		"\"DEWPOINT1_DEGC\":%.2f,"
		"\"DEWPOINT2_DEGC\":%.2f,"
		"\"PWMFAN_DUTY_PCT\":%.2f,"
		"\"PWMFAN_PULSE_HZ\":%.2f,"
		"\"AXIALFAN_STAT_A\":%.2f,"
		"\"MAKEUP_PUMP_CONTROL_V\":%.2f,"
		"\"MAKEUP_PUMP_STAT_A\":%.2f,"
		"\"CIRC_PUMP_CONTROL_V\":%.2f,"
		"\"CIRC_PUMP_STAT_A\":%.2f,"
		"\"HEATER_STAT_A\":%.2f,"
		"\"H2SOLVALVE_1_STAT_A\":%.2f,"
		"\"H2SOLVALVE_2_STAT_A\":%.2f,"
		"\"H2SOLVALVE_3_STAT_A\":%.2f,"
		"\"PSU48_VOLTAGE_SV_V\":%.2f,"
		"\"PSU48_CURRENT_SV_A\":%.2f,"
		"\"ELECTROLYTE_TEMP_SV_DEGC\":%.2f,"
		"\"ELECTROLYTE_FLOW_SV_SLM\":%.2f"
		"} ,\"DI\":{"
		"\"TANK_PRESS_SW\":%s,"
		"\"LEVEL_SINGLE\":%s,"
		"\"LEVEL_DOUBLE_HIGHER\":%s,"
		"\"LEVEL_DOUBLE_LOWER\":%s,"
		"\"HEATER_CTRL\":%s,"
		"\"H2SOLVALVE_1_CTRL\":%s,"
		"\"H2SOLVALVE_2_CTRL\":%s,"
		"\"H2SOLVALVE_3_CTRL\":%s,"
		"\"PSU48_MC_CTRL\":%s,"
		"\"PSU48_MC_STAT\":%s,"
		"\"PSU48_PWR_ENA_CTRL\":%s,"
		"\"PSU48_MODE_CV_CTRL\":%s,"
		"\"PSU48_MODE_CC_CTRL\":%s,"
		"\"ELECTROLYTE_CTRL_CTRL\":%s,"
		"\"HIGHP_PRODUCTION_CTRL\":%s,"
		"\"EMERGENCY_ACTIVE\":%s"
		"}}}",
		HX2K_SCHEMA_STR, HX2K_DEVICE_ID_STR, f->uuid, ts, (unsigned long)f->seq,
		(unsigned long)s_desired_shadow.cmd_rev, ai[0], ai[1], ai[2], ai[3],
		ai[4], ai[5], ai[6], ai[7], ai[8], ai[9], ai[10], ai[11], ai[12],
		ai[13], ai[14], ai[15], ai[16], ai[17], ai[18], ai[19], ai[20], ai[21],
		ai[22], ai[23], ai[24], ai[25], ai[26], ai[27], ai[28], ai[29], ai[30],
		ai[31], di_str[0], di_str[1], di_str[2], di_str[3], di_str[4],
		di_str[6], di_str[7], di_str[8], di_str[9], di_str[10], di_str[11],
		di_str[12], di_str[13], di_str[14], di_str[15], di_str[16]);

	if (len <= 0 || len >= (int)sizeof(json)) {
		ESP_LOGE(TAG, "Telemetry JSON build error or truncated (len=%d)", len);
		return;
	}

	// 토픽: hxb2k/<device_id>/telemetry
	char topic[64];
	snprintf(topic, sizeof(topic), "hxb2k/%s-%s/telemetry", HX2K_DEVICE_ID_STR,
			 f->uuid);

	int msg_id = esp_mqtt_client_publish(client, topic,
										 json, // null terminated
										 0,	   // len = 0 → strlen(json)
										 0,	   // QoS 0
										 0	   // retain 0
	);
	hxb2_tlm_gpio_toggle();
	/*for (int i = 0; i < 8; i++) {
		ESP_LOGI("MQTT", "adc3[%d] = %d", i, f->adc3[i]);
	}*/
	mqtt_device_topic_subscribe();
}

// uart 설정
void uart_init(void) {
	// uart 옵션
	const uart_config_t uart_config = {
		.baud_rate = 115200,
		.data_bits = UART_DATA_8_BITS,
		.parity = UART_PARITY_DISABLE,
		.stop_bits = UART_STOP_BITS_1,
		.flow_ctrl = UART_HW_FLOWCTRL_DISABLE,
		.source_clk = UART_SCLK_APB,
	};

	// 버퍼 설정
	uart_driver_install(UART_NUM_1, RX_BUF_SIZE, 0, 0, NULL, 0);
	// 업션 설정
	uart_param_config(UART_NUM_1, &uart_config);
	// uart 핀 설정
	uart_set_pin(UART_NUM_1, TXD0_PIN, RXD0_PIN, UART_PIN_NO_CHANGE,
				 UART_PIN_NO_CHANGE);
}

static void echo_task(void *arg) {
	uart_init();

	uint8_t rx_buf[256];
	char line_buf[512];
	size_t line_len = 0;

	while (1) {
		int len = uart_read_bytes(UART_NUM_1, rx_buf, sizeof(rx_buf) - 2,
								  1000 / portTICK_PERIOD_MS);

		if (len > 0) {
			for (int i = 0; i < len; i++) {
				char c = (char)rx_buf[i];

				if (c == '\r') {
					// 무시 (CRLF 중 CR)
					continue;
				} else if (c == '\n') {
					// 한 줄 완성
					if (line_len > 0) {
						line_buf[line_len] = '\0';

						hxb2_telemetry_t frame;
						ESP_LOGI("RAW", "%s", line_buf);
						if (parse_stm32_tlm_line(line_buf, &frame)) {
							// 파싱 성공 → 텔레메트리 MQTT publish
							if (g_mqtt_client) {
								hx2k_publish_telemetry(g_mqtt_client, &frame);
							}
						} else {
							ESP_LOGW(TAG, "Parse failed");
						}
						line_len = 0;
					}
				} else {
					if (line_len < sizeof(line_buf) - 1) {
						line_buf[line_len++] = c;
					} else {
						// 라인 오버플로 → 버리고 다시 시작
						ESP_LOGW(TAG, "Line buffer overflow, dropping line");
						line_len = 0;
					}
				}
			}
		} else if (len < 0) {
			ESP_LOGE(TAG, "uart_read_bytes error: %d", len);
		}
	}
}
/** Event handler for Ethernet events */
static void eth_event_handler(void *arg, esp_event_base_t event_base,
							  int32_t event_id, void *event_data) {
	uint8_t mac_addr[6] = {0};
	/* we can get the ethernet driver handle from event data */
	esp_eth_handle_t eth_handle = *(esp_eth_handle_t *)event_data;

	switch (event_id) {
	case ETHERNET_EVENT_CONNECTED:
		esp_eth_ioctl(eth_handle, ETH_CMD_G_MAC_ADDR, mac_addr);
		ESP_LOGI(TAG, "Ethernet Link Up");
		ESP_LOGI(TAG, "Ethernet HW Addr %02x:%02x:%02x:%02x:%02x:%02x",
				 mac_addr[0], mac_addr[1], mac_addr[2], mac_addr[3],
				 mac_addr[4], mac_addr[5]);
		break;
	case ETHERNET_EVENT_DISCONNECTED:
		ESP_LOGI(TAG, "Ethernet Link Down");
		eth_get = false;
		break;
	case ETHERNET_EVENT_START:
		ESP_LOGI(TAG, "Ethernet Started");
		break;
	case ETHERNET_EVENT_STOP:
		ESP_LOGI(TAG, "Ethernet Stopped");
		break;
	default:
		break;
	}
}
void hxb2_mqtt_start(void);
/** Event handler for IP_EVENT_ETH_GOT_IP */
static void got_ip_event_handler(void *arg, esp_event_base_t event_base,
								 int32_t event_id, void *event_data) {
	ip_event_got_ip_t *event = (ip_event_got_ip_t *)event_data;
	const esp_netif_ip_info_t *ip_info = &event->ip_info;

	ESP_LOGI(TAG, "Ethernet Got IP Address");
	ESP_LOGI(TAG, "~~~~~~~~~~~");
	ESP_LOGI(TAG, "ETHIP:" IPSTR, IP2STR(&ip_info->ip));
	ESP_LOGI(TAG, "ETHMASK:" IPSTR, IP2STR(&ip_info->netmask));
	ESP_LOGI(TAG, "ETHGW:" IPSTR, IP2STR(&ip_info->gw));
	ESP_LOGI(TAG, "~~~~~~~~~~~");
	vTaskDelay(30 / portTICK_PERIOD_MS);
	eth_get = true;
	hxb2_mqtt_start();
}
void hxb2_tlm_gpio_init(void) {
	gpio_config_t io_conf = {
		.pin_bit_mask = 1ULL << 2,
		.mode = GPIO_MODE_OUTPUT,
		.pull_up_en = GPIO_PULLUP_DISABLE,
		.pull_down_en = GPIO_PULLDOWN_DISABLE,
		.intr_type = GPIO_INTR_DISABLE,
	};
	gpio_config(&io_conf);

	gpio_set_level(GPIO_NUM_2, 0); // 초기값 Low
}
void app_main(void) {
	// Initialize Ethernet driver
	uint8_t eth_port_cnt = 0;
	esp_eth_handle_t *eth_handles;

	xTaskCreate(echo_task, "uart_echo_task", 4096, NULL, 10, NULL);
	hxb2_tlm_gpio_init();
	esp_log_level_set("emac", ESP_LOG_DEBUG);
	esp_log_level_set("esp.emac", ESP_LOG_DEBUG);
	esp_log_level_set("esp_eth", ESP_LOG_DEBUG);

	ESP_ERROR_CHECK(example_eth_init(&eth_handles, &eth_port_cnt));

	// Initialize TCP/IP network interface aka the esp-netif (should be called
	// only once in application)
	ESP_ERROR_CHECK(esp_netif_init());
	// Create default event loop that running in background
	ESP_ERROR_CHECK(esp_event_loop_create_default());

	esp_netif_t *eth_netifs[eth_port_cnt];
	esp_eth_netif_glue_handle_t eth_netif_glues[eth_port_cnt];

	// Create instance(s) of esp-netif for Ethernet(s)
	if (eth_port_cnt == 1) {
		// Use ESP_NETIF_DEFAULT_ETH when just one Ethernet interface is used
		// and you don't need to modify default esp-netif configuration
		// parameters.
		esp_netif_config_t cfg = ESP_NETIF_DEFAULT_ETH();
		eth_netifs[0] = esp_netif_new(&cfg);
		eth_netif_glues[0] = esp_eth_new_netif_glue(eth_handles[0]);
		// Attach Ethernet driver to TCP/IP stack
		ESP_ERROR_CHECK(esp_netif_attach(eth_netifs[0], eth_netif_glues[0]));
	} else {
		// Use ESP_NETIF_INHERENT_DEFAULT_ETH when multiple Ethernet interfaces
		// are used and so you need to modify esp-netif configuration parameters
		// for each interface (name, priority, etc.).
		esp_netif_inherent_config_t esp_netif_config =
			ESP_NETIF_INHERENT_DEFAULT_ETH();
		esp_netif_config_t cfg_spi = {.base = &esp_netif_config,
									  .stack = ESP_NETIF_NETSTACK_DEFAULT_ETH};
		char if_key_str[10];
		char if_desc_str[10];
		char num_str[3];
		for (int i = 0; i < eth_port_cnt; i++) {
			itoa(i, num_str, 10);
			strcat(strcpy(if_key_str, "ETH_"), num_str);
			strcat(strcpy(if_desc_str, "eth"), num_str);
			esp_netif_config.if_key = if_key_str;
			esp_netif_config.if_desc = if_desc_str;
			esp_netif_config.route_prio -= i * 5;
			eth_netifs[i] = esp_netif_new(&cfg_spi);
			eth_netif_glues[i] = esp_eth_new_netif_glue(eth_handles[i]);
			// Attach Ethernet driver to TCP/IP stack
			ESP_ERROR_CHECK(
				esp_netif_attach(eth_netifs[i], eth_netif_glues[i]));
		}
	}

	// Register user defined event handlers
	ESP_ERROR_CHECK(esp_event_handler_register(ETH_EVENT, ESP_EVENT_ANY_ID,
											   &eth_event_handler, NULL));
	ESP_ERROR_CHECK(esp_event_handler_register(IP_EVENT, IP_EVENT_ETH_GOT_IP,
											   &got_ip_event_handler, NULL));

	// Start Ethernet driver state machine
	for (int i = 0; i < eth_port_cnt; i++) {
		ESP_ERROR_CHECK(esp_eth_start(eth_handles[i]));
	}

	// 1) DHCP client 끄기
	ESP_ERROR_CHECK(esp_netif_dhcpc_stop(eth_netifs[0]));

	// 2) IP 정보 세팅
	esp_netif_ip_info_t ip_info;
	memset(&ip_info, 0, sizeof(ip_info));

	// 장치: 192.168.121.2/24
	IP4_ADDR(&ip_info.ip, 192, 168, 121, 2);
	IP4_ADDR(&ip_info.netmask, 255, 255, 255, 0);

	// 게이트웨이 비움 (ICD에서 공란으로 두라고 해서 0.0.0.0)
	IP4_ADDR(&ip_info.gw, 0, 0, 0, 0);

	ESP_ERROR_CHECK(esp_netif_set_ip_info(eth_netifs[0], &ip_info));

#if CONFIG_EXAMPLE_ETH_DEINIT_AFTER_S >= 0
	// For demonstration purposes, wait and then deinit Ethernet network
	vTaskDelay(pdMS_TO_TICKS(CONFIG_EXAMPLE_ETH_DEINIT_AFTER_S * 1000));
	ESP_LOGI(TAG, "stop and deinitialize Ethernet network...");
	// Stop Ethernet driver state machine and destroy netif
	for (int i = 0; i < eth_port_cnt; i++) {
		ESP_ERROR_CHECK(esp_eth_stop(eth_handles[i]));
		ESP_ERROR_CHECK(esp_eth_del_netif_glue(eth_netif_glues[i]));
		esp_netif_destroy(eth_netifs[i]);
	}
	esp_netif_deinit();
	ESP_ERROR_CHECK(example_eth_deinit(eth_handles, eth_port_cnt));
	ESP_ERROR_CHECK(esp_event_handler_unregister(IP_EVENT, IP_EVENT_ETH_GOT_IP,
												 got_ip_event_handler));
	ESP_ERROR_CHECK(esp_event_handler_unregister(ETH_EVENT, ESP_EVENT_ANY_ID,
												 eth_event_handler));
	ESP_ERROR_CHECK(esp_event_loop_delete_default());
#endif // EXAMPLE_ETH_DEINIT_AFTER_S > 0
}
